import re
import functools
from resources.lib import config, common, scrapers, player, cache, cleanstring, ga

actions = []
def _action(func):
    '''Decorator
    Mark the function as a valid action by
    putting the name of `func` into `actions`
    '''
    actions.append(func.__name__)
    return func

def _dir_action(func):
    '''Decorator
    Assumes `func` returns list of diritems
    Results from calling `func` are used to build plugin directory
    '''
    @_action # Note: must keep this order to get func name
    @functools.wraps(func)
    def make_dir(*args, **kargs):
        diritems = func(*args, **kargs)
        for di in diritems:
            common.add_item(di)
        common.end_dir()
    return make_dir

def _ga_track(action_name, label=None):
    ga.event('Action', action_name, label)

@_dir_action
def index():
    _ga_track('Index')
    return config.index_items

@_dir_action
def shows(url):
    _ga_track('Shows', '%s - Page %s' % scrapers.category_page(url))
    di_list = []
    for eng_name, ori_name, show_url, image in scrapers.shows(url):
        action_url = common.action_url('episodes', url=show_url)
        name = cleanstring.show(eng_name, ori_name)
        di_list.append(common.diritem(name, action_url, image))
    for page, page_url in scrapers.pages(url):
        action_url = common.action_url('shows', url=page_url)
        page_label = cleanstring.page(page)
        di_list.append(common.diritem(page_label, action_url))
    return di_list

@_dir_action
def recent_updates(url):
    _ga_track('Recent updates')
    di_list = []
    for name, update_url in scrapers.recent_updates(url):
        action_url = common.action_url('mirrors', url=update_url)
        di_list.append(common.diritem(name, action_url))
    return di_list

@_dir_action
def episodes(url):
    _ga_track('Episodes', scrapers.show_name(url))
    episodes = scrapers.episodes(url)
    if len(episodes) > 0:
        di_list = []
        for name, episode_url in episodes:
            action_url = common.action_url('mirrors', url=episode_url)
            epi = cleanstring.episode(name)
            di_list.append(common.diritem(epi, action_url))
        return di_list
    else:
        # show mirrors directly if there is no episode listing, e.g. movie
        label, href = scrapers.watch_button(url)
        return _mirrors(href)

@_action
def play_mirror(url):
    with common.busy_indicator:
        vidurl = common.resolve(url)
        if vidurl:
            try:
                title, image = scrapers.title_image(url)
            except Exception:
                # we can proceed without the title and image
                title, image = ('', '')
            _ga_track('Play', title)
            player.play(vidurl, title, image)

@_dir_action
def mirrors(url):
    return _mirrors(url)

def _mirrors(url):
    mirrors = scrapers.mirrors(url)
    num_mirrors = len(mirrors)
    if num_mirrors > 0:
        di_list = []
        for mirr_label, parts in mirrors:
            for part_label, part_url in parts:
                label = cleanstring.mirror(mirr_label, part_label)
                action_url = common.action_url('play_mirror', url=part_url)
                di_list.append(common.diritem(label, action_url, isfolder=False))
        return di_list
    else:
        # if no mirror listing, try to resolve this page directly
        action_url = common.action_url('play_mirror', url=url)
        return [common.diritem(33101, action_url, isfolder=False)]
