# -*- coding: utf-8 -*-
#
#    Copyright (c) 2015 Billy2011, MediaPortal Team
#
# Copyright (C) 2009-2010 Fluendo, S.L. (www.fluendo.com).
# Copyright (C) 2009-2010 Marc-Andre Lureau <marcandre.lureau@gmail.com>

# This file may be distributed and/or modified under the terms of
# the GNU General Public License version 2 as published by
# the Free Software Foundation.
# This file is distributed without any warranty; without even the implied
# warranty of merchantability or fitness for a particular purpose.
# See "LICENSE" in the source distribution for more information.

from itertools import ifilter
import logging
import os, os.path
import tempfile
import urlparse
from cookielib import CookieJar
from Crypto.Cipher import AES

from enigma import eBackgroundFileEraser

from twisted.python import log
from twisted.web import client
from twisted.internet import defer, reactor, task
from twisted.internet.task import deferLater
from twisted.web.http_headers import Headers
from twagenthelper import twAgentGetPage
from debuglog import printlog as printl

from m3u8 import M3U8

class HLSFetcher(object):

	def __init__(self, url, options=None, program=1, headers=None):
		if '|X-Forwarded-For=' in url:
			url, header_val = url.split('|X-Forwarded-For=')
			headers = {'X-Forwarded-For':[header_val]}
		self.url = url
		self.hls_headers = Headers(headers)
		self.program = program
		if options:
			self.path = options.get('path',None)
			self.referer = options.get('referer',None)
			self.bitrate = options.get('bitrate',200000)
			self.nbuffer = options.get('buffer',3)
			self.n_segments_keep = options.get('keep',self.nbuffer+1)
		else:
			self.path = None
			self.referer = None
			self.bitrate = 200000
			self.n_segments_keep = 3
			self.nbuffer = 4
		if not self.path:
			self.path = tempfile.mkdtemp()

		self._program_playlist = None
		self._file_playlist = None
		self._cookies = CookieJar()
		self._cached_files = {} 	# sequence n -> path
		self._run = True

		self._files = None 			# the iter of the playlist files download
		self._next_download = None 	# the delayed download defer, if any
		self._file_playlisted = None # the defer to wait until new files are added to playlist
		self._new_filed = None

		if self.referer:
			self.hls_headers.setRawHeaders('Referer', [self.referer])

	def _get_page(self, url):
		def got_page(content):
			return content
		def got_page_error(e):
			return e

		url = url.encode("utf-8")
		if 'HLS_RESET_COOKIES' in os.environ.keys():
			self._cookies.clear()

		if not '.m3u8' in url:
			timeout = 15
		else:
			timeout = 60

		headers = Headers(self.hls_headers._rawHeaders.copy())
		d = twAgentGetPage(url, cookieJar=self._cookies, headers=headers, timeout=timeout)
		d.addCallback(got_page)
		d.addErrback(got_page_error)
		return d

	def delete_cache(self, f):
		keys = self._cached_files.keys()
		for i in ifilter(f, keys):
			filename = self._cached_files[i]
			os.remove(filename)
			del self._cached_files[i]

	def clear_all_cache(self):
		bgFileEraser = eBackgroundFileEraser.getInstance()
		for filename in self._cached_files.itervalues():
			bgFileEraser.erase(str(filename))
		self._cached_files.clear()

	def _got_file_failed(self, e):
		if self._new_filed:
			self._new_filed.errback(e)
			self._new_filed = None

	def _got_file(self, path, url, f):
		self._cached_files[f['sequence']] = path
		if self.n_segments_keep != -1:
			self.delete_cache(lambda x: x <= f['sequence'] - self.n_segments_keep)
		if self._new_filed:
			self._new_filed.callback((path, url, f))
			self._new_filed = None
		return (path, url, f)

	def _download_page(self, url, path):
		# client.downloadPage does not support cookies!
		def _check(x):
			if not self._run:
				raise Exception('Download cancelled.')
			else:
				return x

		d = self._get_page(url)
		f = open(path, 'wb')
		d.addCallback(_check)
		if self._file_playlist._key and self._file_playlist._iv:
			aes = AES.new(self._file_playlist._key, AES.MODE_CBC, self._file_playlist._iv)
			d.addCallback(lambda x: f.write(aes.decrypt(x)))
		else:
			d.addCallback(lambda x: f.write(x))
		d.addBoth(lambda _: f.close())
		d.addCallback(lambda _: path)
		return d

	def _download_file(self, f):
		l = make_url(self._file_playlist.url, f['file'])
		name = next(tempfile._get_candidate_names())
		path = os.path.join(self.path, name)
		d = self._download_page(l, path)
		d.addCallback(self._got_file, l, f)
		return d

	def _hasCacheSeq(self, last_file):
		for i in range(1, self.nbuffer):
			if self._cached_files.has_key(last_file['sequence'] - i):
				return True
		return False

	def _get_next_file(self, last_file=None):
		next = self._files.next()
		if next:
			delay = 0
			if last_file:
				if self.nbuffer > 0 and not self._hasCacheSeq(last_file):
					delay = 0
				elif self._file_playlist.endlist():
					delay = 1
				else:
					delay = last_file['duration'] * 0.5	# doesn't work
														# when duration is not in sync with
														# player, which can happen easily...
			return deferLater(reactor, delay, self._download_file, next)
		elif not self._file_playlist.endlist():
			self._file_playlisted = defer.Deferred()
			self._file_playlisted.addCallback(lambda x: self._get_next_file(last_file))
			return self._file_playlisted

	def _handle_end(self, failure):
		failure.trap(StopIteration)
		print "End of media"

	def _get_files_loop(self, last_file=None):
		if self._run:
			if last_file:
				(path, l, f) = last_file
			else:
				f = None
			d = self._get_next_file(f)
			# and loop
			d.addCallback(self._get_files_loop)
			d.addErrback(self._handle_end)

	def _playlist_updated(self, pl):
		if pl and pl.has_programs():
			# if we got a program playlist, save it and start a program
			self._program_playlist = pl
			(program_url, _) = pl.get_program_playlist(self.program, self.bitrate)
			l = make_url(self.url, program_url)
			return self._reload_playlist(M3U8(l))
		elif pl and pl.has_files():
			# we got sequence playlist, start reloading it regularly, and get files
			self._file_playlist = pl
			if not self._files:
				self._files = pl.iter_files()
			if not pl.endlist():
				reactor.callLater(pl.reload_delay(), self._reload_playlist, pl)
			if self._file_playlisted:
				self._file_playlisted.callback(pl)
				self._file_playlisted = None
		else:
			s = 'Playlist has no valid content.'
			printl(s,self,'E')
			raise Exception(s)
		return pl

	def _got_playlist_content(self, content, pl):
		if self._run and not pl.update(content):
			# if the playlist cannot be loaded, start a reload timer
			d = deferLater(reactor, pl.reload_delay(), self._fetch_playlist, pl)
			d.addCallback(self._got_playlist_content, pl)
			return d
		return pl

	def _fetch_playlist(self, pl):
		d = self._get_page(pl.url)
		return d

	def _reload_playlist(self, pl):
		if self._run:
			d = self._fetch_playlist(pl)
			d.addCallback(self._got_playlist_content, pl)
			d.addCallback(self._playlist_updated)
			return d
		else:
			return None

	def get_file(self, sequence):
		d = defer.Deferred()
		keys = self._cached_files.keys()
		try:
			sequence = ifilter(lambda x: x >= sequence, keys).next()
			filename = self._cached_files[sequence]
			d.callback(filename)
		except:
			d.addCallback(lambda x: self.get_file(sequence))
			self._new_filed = d
			keys.sort()
		return d

	def start(self):
		self._files = None
		d = self._reload_playlist(M3U8(self.url))
		d.addCallback(lambda _: self._get_files_loop())
		self._new_filed = defer.Deferred()
		return self._new_filed

	def stop(self):
		self._run = False
		if self._new_filed != None:
			self._new_filed.cancel()
		reactor.callLater(5, self.clear_all_cache)

def make_url(base_url, url):
	if urlparse.urlsplit(url).scheme == '':
		url = urlparse.urljoin(base_url, url)
	if 'HLS_PLAYER_SHIFT_PORT' in os.environ.keys():
		shift = int(os.environ['HLS_PLAYER_SHIFT_PORT'])
		p = urlparse.urlparse(url)
		loc = p.netloc
		if loc.find(":") != -1:
			loc, port = loc.split(':')
			port = int(port) + shift
			loc = loc + ":" + str(port)
		elif p.scheme == "http":
			port = 80 + shift
			loc = loc + ":" + str(shift)
		p = urlparse.ParseResult(scheme=p.scheme,
								netloc=loc,
								path=p.path,
								params=p.params,
								query=p.query,
								fragment=p.fragment)
		url = urlparse.urlunparse(p)
	return url
