# -*- coding: utf-8 -*-
#    HLS-Player for MediaPortal
#
#    Copyright (c) 2015 Billy2011, MediaPortal Team
#
# Copyright (C) 2009-2010 Fluendo, S.L. (www.fluendo.com).
# Copyright (C) 2009-2010 Marc-Andre Lureau <marcandre.lureau@gmail.com>
# Copyright (C) 2010 Zaheer Abbas Merali  <zaheerabbas at merali dot org>
# Copyright (C) 2010 Andoni Morales Alastruey <ylatuya@gmail.com>
# Copyright (C) 2014 Juan Font Alonso <juanfontalonso@gmail.com>

# This file may be distributed and/or modified under the terms of
# the GNU General Public License version 2 as published by
# the Free Software Foundation.
# This file is distributed without any warranty; without even the implied
# warranty of merchantability or fitness for a particular purpose.
# See "LICENSE" in the source distribution for more information.

import sys
import os
import argparse
import Queue
import re

from twisted.web import server, resource, http
from twisted.web.server import NOT_DONE_YET
from twisted.internet import reactor, defer
from twisted.web.static import File
from twisted.web.client import getPage

from fetcher import HLSFetcher
from m3u8 import M3U8

from BaseHTTPServer import HTTPServer, BaseHTTPRequestHandler
from SocketServer import ThreadingMixIn
import threading
from functools import partial
import urlparse

PROXY_PORT = 9080

try:
	from Components.ServiceEventTracker import ServiceEventTracker
	from Components.config import config
	from enigma import eBackgroundFileEraser, iPlayableService, eTimer
	import mp_globals
	import simpleplayer
	from simpleplayer import SimplePlayer
	from debuglog import printlog as printl
except:
	print 'Imports error: isDream: False'
	isDream = False
else:
	isDream = True
	print 'Imports: isDream: True'

CHUNK_SZ = 4096

class HLSControler:

	def __init__(self, fetcher=None):
		self.fetcher = fetcher
		self.player = None

		self._player_sequence = None
		self._n_segments_keep = None

	def set_player(self, player):
		self.player = player
		if player:
			self.player.connect_about_to_finish(self.on_player_about_to_finish)
			self._n_segments_keep = self.fetcher.n_segments_keep
			self.fetcher.n_segments_keep = -1

	def _start(self, first_file):
		(path, l, f) = first_file
		self._player_sequence = f['sequence']
		if self.player:
			self.player.set_uri(path)
			self.player.play()

	def start(self):
		if self.fetcher:
			d = self.fetcher.start()
			d.addCallback(self._start)

	def _set_next_uri(self):
		# keep only the past three segments
		if self._n_segments_keep != -1:
			self.fetcher.delete_cache(lambda x:
				x <= self._player_sequence - self._n_segments_keep)
		self._player_sequence += 1
		print '_set_next_uri:',self._player_sequence
		d = self.fetcher.get_file(self._player_sequence)
		d.addCallback(self.player.set_uri)

	def on_player_about_to_finish(self):
		reactor.callFromThread(self._set_next_uri)

	def clientFinished(self, result, err=False):
		print 'clientFinished:',result
		self.fetcher.stop()
		self.fetcher = None
		self.player = None

class GSTPlayer:

	def __init__(self, request):
		self._playing = False
		self._cb = None
		self._request = request
		self._streamqueue = Queue.Queue()
		self.filepath = None
		self.playerReady = True
		self.bytes_sent = 0

	def play(self):
		#self._playing = True
		print 'Play'

	def stop(self):
		self._playing = False
		print 'Stop'

	def set_uri(self, filepath):
		print 'set_uri:',filepath
		if self._streamqueue:
			self._streamqueue.put(filepath)
			sz = self._streamqueue.qsize()
			if self.playerReady and self._playing:
				reactor.callFromThread(self.sendSegm)
			elif sz > 1:
				self._playing = True
				reactor.callFromThread(self.sendSegm)
			else:
				self._on_about_to_finish()
		else:
			self.filepath = filepath
			self.sendSegm()

	def sendSegm(self):
		sent = 0
		while self.playerReady and sent < 2 and not self._streamqueue.empty():
			self.playerReady = False
			filepath = self._streamqueue.get()
			try:
				size = os.path.getsize(filepath)
			except IOError, e:
				raise Exception(e)
			else:
				print 'sendSegm:',filepath,size
				with open(filepath, 'rb') as f:
					count = 0
					for chunk in iter(partial(f.read, CHUNK_SZ), ''):
						if not self._request._disconnected:
							self._request.write(chunk)
							count += CHUNK_SZ
						else:
							self.stop()
							self._request.finish()
							return NOT_DONE_YET
		
				if count >= (size // 1):
					self.playerReady = True
					sent += 1
					self.bytes_sent += size
		self._on_about_to_finish()

	def _on_about_to_finish(self, p=None):
		if self._cb:
			self._cb()

	def connect_about_to_finish(self, cb):
		self._cb = cb

class HLSProxy(resource.Resource):
	isLeaf = True

	def render_GET(self, request):
		print'render_GET:',str(request.args)
		for header, value in request.requestHeaders.getAllRawHeaders():
			print header, value

		options = self.getOptions(request)
		d = defer.succeed(request)
		d.addCallback(self.getUrl)
		d.addCallback(self.setupHLSPlayer, request, options)
		d.addErrback(self.reqError, request)
		return NOT_DONE_YET

	def setupHLSPlayer(self, url, request, options):
		if url:
			print 'setupHLSPlayer:',url
			request.setHeader('Content-Type', 'video/MP2T')
			c = HLSControler(HLSFetcher(url, options=options))
			p = GSTPlayer(request)
			request.notifyFinish().addCallback(c.clientFinished).addErrback(c.clientFinished, True)
			c.set_player(p)
			c.start()
		else:
			raise Exception("No HLS-URL in request args!")

	def getOptions(self, request):
		options = {}
		if 'bitrate' in request.args:
			options['bitrate'] = int(request.args['bitrate'][0])
			print 'bitrate:',options['bitrate']
		if 'path' in request.args:
			options['path'] = request.args['path'][0]
		if 'referer' in request.args:
			options['referer'] = request.args['referer'][0]
		if 'keep' in request.args:
			options['keep'] = int(request.args['keep'][0])
		if 'buffer' in request.args:
			options['buffer'] = int(request.args['buffer'][0])
		return options

	def getUrl(self, request):
		try:
			url = request.args['url'][0]
		except:
			raise Exception('No HLS-URL in request args!')
		else:
			if url.startswith('newtopia-stream'):
				return self.startNTStream()
			else:
				return url

	def startNTStream(self):
		auth_url = 'https://token.newtopia.de/api/2/GetLongLivedAuthToken/?callback=&authToken=&_='
		return getPage(auth_url, timeout=10).addCallback(self.getNTAuth).addErrback(self.getNTAuth, True)

	def getNTAuth(self, jdata, err=False):
		try:
			auth_token = re.search('authToken":"(.*?)"', jdata).group(1)
			stream_url = 'https://token.newtopia.de/api/2/GetToken/?callback=&authToken=%s&streamKey=stream_live_1&platform=web&_=' % auth_token
		except:
			raise Exception('Cannot get newtopia authToken!')
		else:
			return getPage(stream_url, timeout=10).addCallback(self.getNTM3U8URL).addErrback(self.getNTM3U8URL, True)

	def getNTM3U8URL(self, jdata):
		try:
			url = re.search('"url":"(.*?)"', jdata).group(1)
		except:
			raise Exception('Cannot get newtopia stream url!')
		else:
			return url

	def reqError(self, err, request):
		print '[HLSProxy] Error:',err
		request.setResponseCode(400)
		request.write(str(err))
		request.finish()
		return NOT_DONE_YET

site = server.Site(HLSProxy())

if isDream:
	
	server = reactor.listenTCP(mp_globals.hls_proxy_port, site)
	mp_globals.hls_proxy_port = server.getHost().port
	print 'Startet HLS-Proxy on port '+str(mp_globals.hls_proxy_port)
	
	class m3u8Player(SimplePlayer):

		def __init__(self, session, playlist, playIdx=0):
			self.bufferSeconds = 0.0
			self.bufferPercent = 0.0
			self.bufferSecondsLeft = 0
			self.window = None
			self.endReached = False
			self.duration = 0.0
			self.playPosition = 0.0
			self.bitrate = 0
			self.autoPlay = False

			SimplePlayer.__init__(self, session, playlist, showPlaylist=False, ltype='hls-player', useResume=False, playIdx=playIdx)

		def bufferFull(self):
			if self.autoPlay:
				if self.seekstate != self.SEEK_STATE_PLAY :
					self.setSeekState(self.SEEK_STATE_PLAY)

		def bufferEmpty(self):
			if self.autoPlay:
				if self.seekstate != self.SEEK_STATE_PAUSE :
					self.setSeekState(self.SEEK_STATE_PAUSE)

		def setBuffersize(self):
			service = self.session.nav.getCurrentService()
			if self.bitrate > 2000000:
				service.streamed().setBufferSize(16 * 1024 ** 2)
			else:
				service.streamed().setBufferSize(8 * 1024 ** 2)

		def getVideo(self):
			self.playM3U8(self.playList[self.playIdx][1])

		def playM3U8(self, url):
			self.bitrate = self._getBandwidth()
			path = config.mediaportal.storagepath.value
			url = 'http://127.0.0.1:%d/?url=%s&bitrate=%d&path=%s' % (mp_globals.hls_proxy_port, url, self.bitrate, path)
			title = self.playList[self.playIdx][0]
			self.playStream(title, url)

		def _getBandwidth(self):
			videoPrio = int(config.mediaportal.videoquali_others.value)
			if videoPrio == 2:
				bw = 3000000
			elif videoPrio == 1:
				bw = 1000000
			else:
				bw = 250000

			return bw

		def playExit(self):
			import mp_globals
			mp_globals.yt_tmp_storage_dirty = True
			SimplePlayer.playExit(self)
else:
	server = reactor.listenTCP(PROXY_PORT, site)
	print 'Startet HLS-Proxy on port '+str(PROXY_PORT)
	print 'Use <Ctrl-C> to stop server.'
	reactor.run()


