# -*- coding: UTF-8 -*-
#################################################################################
#
#    SubsSupport 1.2.0 for Enigma2
#    Coded by mx3L (c) 2014
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#################################################################################

from Queue import Queue
import json
from os import path as os_path, listdir
import os
from re import compile as re_compile
import re
from threading import Thread
from twisted.internet.defer import Deferred
from twisted.web import client

from Components.ActionMap import ActionMap, NumberActionMap, HelpableActionMap
from Components.ConfigList import ConfigListScreen
from Components.FileList import FileList
from Components.GUIComponent import GUIComponent
from Components.Harddisk import harddiskmanager
from Components.Label import Label
from Components.Language import language
from Components.MenuList import MenuList
from Components.MultiContent import MultiContentEntryText, \
    MultiContentEntryPixmapAlphaTest
from Components.Renderer.Renderer import Renderer
from Components.ServiceEventTracker import ServiceEventTracker, InfoBarBase
from Components.Sources.List import List
from Components.Sources.StaticText import StaticText
from Components.config import ConfigSubsection, ConfigSelection, ConfigYesNo, \
    configfile, getConfigListEntry, config, ConfigText, ConfigDirectory, ConfigOnOff, \
    ConfigNothing, ConfigInteger, NoSave, KEY_DELETE, KEY_BACKSPACE, KEY_ASCII, \
    KEY_TIMEOUT
from Screens.ChoiceBox import ChoiceBox
from Screens.InfoBarGenerics import InfoBarSeek
from Screens.LocationBox import LocationBox
from Screens.MessageBox import MessageBox
from Screens.Screen import Screen
from Tools.Directories import SCOPE_SKIN_IMAGE, SCOPE_SKIN, resolveFilename, \
    fileExists
from Tools.ISO639 import LanguageCodes
from Tools.LoadPixmap import LoadPixmap

from e2_utils import messageCB, E2SettingsProvider, MyLanguageSelection, unrar, \
    ConfigFinalText, Captcha, DelayMessageBox, MyConfigList
from enigma import RT_HALIGN_RIGHT, RT_VALIGN_TOP, eSize, ePoint, RT_HALIGN_LEFT, \
    RT_HALIGN_RIGHT, RT_HALIGN_CENTER, RT_VALIGN_CENTER, eListboxPythonMultiContent, \
    gFont, getDesktop, eServiceCenter, iServiceInformation, eServiceReference, \
    iSeekableService, iPlayableService, iPlayableServicePtr, eTimer, addFont, gFont, \
    gRGB, loadPNG, ePythonMessagePump, eConsoleAppContainer, eLabel
from parsers import SubRipParser, MicroDVDParser
from process import SubsLoader, DecodeError, ParseError, ParserNotFoundError, \
    LoadError
from searchsubs import Messages
from seek import SubsSeeker, SubtitlesDownloadError, SubtitlesErrors
from seekers.utilities import detectSearchParams
from skin import parseColor, parsePosition, parseFont
from utils import toString, SimpleLogger

from . import _, __author__, __version__, __email__


try:
    from xml.etree.cElementTree import parse as parse_xml
except ImportError:
    from xml.etree.ElementTree import parse as parse_xml


try:
    from Screens.AudioSelection import QuickSubtitlesConfigMenu
except ImportError:
    QuickSubtitlesConfigMenu = None



# localization function

def warningMessage(session, text):
    session.open(MessageBox, text, type=MessageBox.TYPE_WARNING, timeout=5)

def debug(text, *args):
    if DEBUG:
        if len(args) == 1 and isinstance(args[0], tuple):
            text = text % args[0]
        else:
            text = text % (args)
        print "[SubsSupport]", text.encode('utf-8')


# set the name of plugin in which this library belongs
# PLUGIN_NAME = 'mediaplayer2'

# set debug mode
DEBUG = False

# set supported encodings, you have to make sure, that you have corresponding python
# libraries in %PYTHON_PATH%/encodings/ (ie. iso-8859-2 requires iso_8859_2.searchsubs library)

# to choose encodings for region you want, visit:
# http://docs.python.org/release/2.4.4/lib/standard-encodings.html

# Common encodings for all languages
ALL_LANGUAGES_ENCODINGS = ['utf-8', 'utf-16']

# other encodings, sorted according usage
CENTRAL_EASTERN_EUROPE_ENCODINGS = ['windows-1250', 'iso-8859-2', 'maclatin2', 'IBM852']
WESTERN_EUROPE_ENCODINGS = ['windows-1252', 'iso-8859-15', 'macroman', 'ibm1140', 'IBM850']
RUSSIAN_ENCODINGS = ['windows-1251', 'cyrillic', 'maccyrillic', 'koi8_r', 'IBM866']
ARABIC_ENCODINGS = ['windows-1256', 'iso-8859-6', 'IBM864']
TURKISH_ENCODINGS = ['windows-1254', 'iso-8859-9', 'latin5', 'macturkish', 'ibm1026', 'IBM857']
GREEK_ENCODINGS = ['windows-1253', 'iso-8859-7', 'macgreek']

ENCODINGS = {("Central and Eastern Europe") : CENTRAL_EASTERN_EUROPE_ENCODINGS,
            ("Western Europe"):WESTERN_EUROPE_ENCODINGS,
            ("Russia"):RUSSIAN_ENCODINGS,
            ("Arabic"): ARABIC_ENCODINGS,
            ("Turkish"):TURKISH_ENCODINGS,
            ("Greek"):GREEK_ENCODINGS}


# fontname-R: regular font - mandatory
# fontname-B: bold font - optional, if missing regular font will be used
# fontname-I: italic font - optional, if missing regular font will be used

E2_FONT_PATH = '/usr/share/fonts/'
FONT_PATH = os.path.join(os.path.dirname(__file__), "fonts")

# we use fonts from local path and enigma2 fonts path
FONTS_LOCAL = [(fontfile, FONT_PATH) for fontfile in os.listdir(FONT_PATH)]
FONTS_E2 = []  # [(fontfile, E2_FONT_PATH) for fontfile in os.listdir(E2_FONT_PATH)]
FONTS = FONTS_LOCAL + FONTS_E2

FONT = {
        "Default":
          {
            "regular":"Regular",
            "italic":"Regular",
            "bold":"Regular",
            "path":"Regular"
          }
        }

RE_FONT = '(.+?)\.ttf$'
RE_FONT_RIB = '(.+?)-(R|I|B)\.ttf$'


# looking for fonts in selected paths
for fontfile, path in FONTS:
    font = re.search(RE_FONT, fontfile)
    font_rib = re.search(RE_FONT_RIB, fontfile)

    if font_rib and (font_rib.group(1) in FONT):
        continue
    if font and (font.group(1) in FONT):
        continue

    if font_rib:
        fontname = font_rib.group(1)
        fonttype = font_rib.group(2)
        regular = fontname + '-R.ttf'
        italic = fontname + '-I.ttf'
        bold = fontname + '-B.ttf'

        if fonttype == 'R':
            FONT[fontname] = {}
            FONT[fontname]["path"] = path
            FONT[fontname]["regular"] = fontfile

            if (italic, path) in FONTS:
                FONT[fontname]["italic"] = italic
            else:
                print "[SubsSupport] missing %s font file" % italic
                FONT[fontname]["italic"] = fontfile
            if (bold, path) in FONTS:
                FONT[fontname]["bold"] = bold
            else:
                print "[SubsSupport] missing %s font file" % bold
                FONT[fontname]["bold"] = fontfile

    elif font:
        FONT[font.group(1)] = {"regular":fontfile,
                               "italic":fontfile,
                               "bold":fontfile,
                               "path":path}

# initializing fonts
print "[SubsSupport] initializing fonts in %s" % FONT_PATH

for f in FONT.keys():
    print "[SubsSupport] initializing %s" % f
    regular = FONT[f]['regular']
    italic = FONT[f]['italic']
    bold = FONT[f]['bold']
    path = FONT[f]['path']

    if f == 'Default':
        continue

    # e2 fonts are already initialized
    if path == E2_FONT_PATH:
        continue

    addFont(os.path.join(path, regular), regular, 100, False)
    addFont(os.path.join(path, italic), italic, 100, False)
    addFont(os.path.join(path, bold), bold, 100, False)

# initializing parsers
PARSERS = [SubRipParser, MicroDVDParser]

def getEmbeddedFontSizeCfg(defaultFontSizeCfg):
    CONFIG_SUBTITLES_OPENPLI = "subtitles"
    CONFIG_FONTSIZE_OPENPLI = "subtitle_fontsize"
    CONFIG_SUBTITLES_VTI = "subtitle"
    CONFIG_FONTSIZE_VTI = "subtitlefontsize"

    try:
        subtitles_pli_cfg = getattr(config, CONFIG_SUBTITLES_OPENPLI)
    except KeyError:
        subtitles_pli_cfg = None
    if subtitles_pli_cfg is not None:
        try:
            return getattr(subtitles_pli_cfg, CONFIG_FONTSIZE_OPENPLI)
        except KeyError:
            pass
    try:
        subtitles_vti_cfg = getattr(config, CONFIG_SUBTITLES_VTI)
    except KeyError:
        subtitles_vti_cfg = None
    if subtitles_vti_cfg is not None:
        try:
            return getattr(subtitles_vti_cfg, CONFIG_FONTSIZE_VTI)
        except KeyError:
            pass
    return defaultFontSizeCfg

GLOBAL_CONFIG_INIT = False

def initSubsSettings(configSubsection=None):
    global GLOBAL_CONFIG_INIT
    if configSubsection:
        print '[SubsSupport] using provided ConfigSubsection to store config'
        subtitles_settings = configSubsection
    elif 'PLUGIN_NAME' in globals():
        print '[SubsSupport] using config.plugins.%s.%s to store config' % (PLUGIN_NAME, 'subtitles')
        plugin_settings = getattr(config.plugins, PLUGIN_NAME)
        setattr(plugin_settings, 'subtitles', ConfigSubsection())
        subtitles_settings = getattr(plugin_settings, 'subtitles')
    elif GLOBAL_CONFIG_INIT:
        print "[SubsSupport] using global config (already initialized)"
        return config.plugins.subtitlesSupport
    else:
        print "[SubsSupport] using global config"
        config.plugins.subtitlesSupport = ConfigSubsection()
        subtitles_settings = config.plugins.subtitlesSupport
        GLOBAL_CONFIG_INIT = True

    print '[SubsSupport] initializing config'

    fontChoiceList = [f for f in FONT.keys()]
    fontSizeChoiceList = [("%d" % i, "%d px" % i) for i in range(10, 60, 1)]
    positionChoiceList = [("0", _("top"))]
    positionChoiceList.extend([("%d" % i, "%d %%" % i) for i in range(1, 100, 1)])
    positionChoiceList.append(("100", _("bottom")))
    shadowSizeChoiceList = [("%d" % i, "%d px" % i) for i in range(1, 8, 1)]
    shadowOffsetChoiceList = [("%d" % i, "%d px" % i) for i in range(-8, -1, 1)]
    backgroundOffsetChoiceList = [("%d" % i, "%d px" % i) for i in range(5, 100, 1)]
    colorChoiceList = []
    colorChoiceList.append(("ff0000", _("red")))
    colorChoiceList.append(("DCDCDC", _("grey")))
    colorChoiceList.append(("00ff00", _("green")))
    colorChoiceList.append(("ff00ff", _("purple")))
    colorChoiceList.append(("ffff00", _("yellow")))
    colorChoiceList.append(("ffffff", _("white")))
    colorChoiceList.append(("00ffff", _("blue")))
    colorChoiceList.append(("000000", _("black")))
    COLORFILE = os.path.join(os.path.dirname(__file__), 'colors.txt')
    print '[SubsSupport] looking for custom colors in', COLORFILE
    try:
        with open(COLORFILE, 'r') as f:
            for line in f:
                color = re.search('^(\w+)\s+([0-9A-Fa-f]{6})$', line)
                if color is not None:
                    alias = color.group(1)
                    hex_color = color.group(2)
                    print '[SubsSupport] adding custom color', alias
                    colorChoiceList.append((hex_color, alias))
    except IOError as e:
        print '[SubsSupport] error while loading custom colors', str(e)

    alphaChoiceList = [("00", _("opaque"))]
    alphaChoiceList.extend([("%02x" % val, "%d %%"%(int(percent * 100 / float(32)))) for percent, val in enumerate(xrange(0, 256, 8)) if val != 0])
    alphaChoiceList.append(("ff", _("transparent")))

    subtitles_settings.pauseVideoOnSubtitlesMenu = ConfigYesNo(default=True)
    subtitles_settings.encodingsGroup = ConfigSelection(default="Central and Eastern Europe", choices=[(e, _(e)) for e in ENCODINGS.keys()])

    subtitles_settings.external = ConfigSubsection()
    subtitles_settings.external.position = ConfigSelection(default="94", choices=positionChoiceList)
    subtitles_settings.external.fontType = ConfigSelection(default="Ubuntu", choices=fontChoiceList)
    subtitles_settings.external.fontSize = ConfigSelection(default="43", choices=fontSizeChoiceList)
    subtitles_settings.external.color = ConfigSelection(default="ffffff", choices=colorChoiceList)
    subtitles_settings.external.alpha = ConfigSelection(default="00", choices=alphaChoiceList)
    subtitles_settings.external.shadow = ConfigSubsection()
    subtitles_settings.external.shadow.enabled = ConfigOnOff(default=True)
    subtitles_settings.external.shadow.type = ConfigSelection(default="border", choices=[("offset", _("offset")), ("border", _('border'))])
    subtitles_settings.external.shadow.color = ConfigSelection(default="000000", choices=colorChoiceList)
    subtitles_settings.external.shadow.size = ConfigSelection(default="2", choices=shadowSizeChoiceList)
    subtitles_settings.external.shadow.xOffset = ConfigSelection(default="-3", choices=shadowOffsetChoiceList)
    subtitles_settings.external.shadow.yOffset = ConfigSelection(default="-3", choices=shadowOffsetChoiceList)
    subtitles_settings.external.background = ConfigSubsection()
    subtitles_settings.external.background.enabled = ConfigOnOff(default=True)
    subtitles_settings.external.background.type = ConfigSelection(default="dynamic", choices=[("dynamic", _("dynamic")), ("static", _("static"))])
    subtitles_settings.external.background.xOffset = ConfigSelection(default="10", choices=backgroundOffsetChoiceList)
    subtitles_settings.external.background.yOffset = ConfigSelection(default="10", choices=backgroundOffsetChoiceList)
    subtitles_settings.external.background.color = ConfigSelection(default="000000", choices=colorChoiceList)
    subtitles_settings.external.background.alpha = ConfigSelection(default="80", choices=alphaChoiceList)

    subtitles_settings.embedded = ConfigSubsection()
    subtitles_settings.embedded.position = ConfigSelection(default="94", choices=positionChoiceList)
    subtitles_settings.embedded.fontType = ConfigSelection(default="Ubuntu", choices=fontChoiceList)
    subtitles_settings.embedded.fontSize = ConfigSelection(default="34", choices=fontSizeChoiceList)
    subtitles_settings.embedded.color = ConfigSelection(default="ffffff", choices=colorChoiceList)
    subtitles_settings.embedded.shadow = ConfigSubsection()
    subtitles_settings.embedded.shadow.size = ConfigSelection(default="3", choices=shadowSizeChoiceList)
    subtitles_settings.embedded.shadow.color = ConfigSelection(default="000000", choices=colorChoiceList)
    subtitles_settings.embedded.shadow.xOffset = ConfigSelection(default="-3", choices=shadowOffsetChoiceList)
    subtitles_settings.embedded.shadow.yOffset = ConfigSelection(default="-3", choices=shadowOffsetChoiceList)

    subtitles_settings.engine = ConfigSubsection()
    subtitles_settings.engine.type = ConfigSelection(default="standard_new", choices=[('standard', _("Block (old)")), ('standard_new', _("Block (new)")), ('extended', _("Row (experimental)"))])
    subtitles_settings.engine.expert = ConfigSubsection()
    subtitles_settings.engine.expert.show = NoSave(ConfigYesNo(default=False))
    subtitles_settings.engine.expert.playerDelay = ConfigSelection(default="0", choices=[("%d" % i, "%d ms" % i) for i in range(0, 20000, 200)])
    subtitles_settings.engine.expert.startDelay = ConfigSelection(default="1200", choices=[("%d" % i, "%d ms" % i) for i in range(0, 100, 1500)])
    subtitles_settings.engine.expert.hideDelay = ConfigSelection(default="200", choices=[("%d" % i, "%d ms" % i) for i in range(0, 1000, 50)])
    subtitles_settings.engine.expert.ptsDelayCheck = ConfigSelection(default="200", choices=[("%d" % i, "%d ms" % i) for i in range(100, 1000, 100)])
    subtitles_settings.engine.expert.syncDelay = ConfigSelection(default="300", choices=[("%d" % i, "%d ms" % i) for i in range(100, 1000, 100)])
    subtitles_settings.engine.expert.refreshDelay = ConfigSelection(default="1000", choices=[("%d" % i, "%d ms" % i) for i in range(200, 3000, 200)])

    subtitles_settings.search = ConfigSubsection()
    subtitles_settings.search.downloadPath = ConfigDirectory(default="/tmp/")
    subtitles_settings.search.tmpPath = ConfigDirectory(default="/tmp/")
    subtitles_settings.search.lang1 = ConfigFinalText(default=language.getLanguage()[:2])
    subtitles_settings.search.lang2 = ConfigFinalText(default=language.getLanguage()[:2])
    subtitles_settings.search.lang3 = ConfigFinalText(default=language.getLanguage()[:2])
    subtitles_settings.search.timeout = ConfigSelection(default="10", choices=[(("%d" % i, "%d s" % i)) for i in range(5, 20)])
    subtitles_settings.search.history = ConfigText(default="")
    subtitles_settings.search.title = ConfigTextWithSuggestionsAndHistory(subtitles_settings.search.history, default="", fixed_size=False)
    subtitles_settings.search.type = ConfigSelection(default="movie", choices=[("tv_show", _("TV show")), ("movie", _("Movie"))])
    subtitles_settings.search.year = ConfigInteger(default=0, limits=(0, 2100))
    subtitles_settings.search.season = ConfigInteger(default=0, limits=(0, 100))
    subtitles_settings.search.episode = ConfigInteger(default=0, limits=(0, 100))
    subtitles_settings.search.provider = ConfigSelection(default="all", choices=[("all", _("All")), ])
    subtitles_settings.search.movieProvider = ConfigSelection(default="all", choices=[("all", _("All")), ])
    subtitles_settings.search.tvshowProvider = ConfigSelection(default="all", choices=[("all", _("All")), ])
    subtitles_settings.search.useFilePath = ConfigYesNo(default=True)
    subtitles_settings.search.manualSearch = ConfigYesNo(default=False)
    subtitles_settings.search.defaultSort = ConfigSelection(default='lang', choices=[('lang', _("Language")), ('provider', _("Provider"))])
    subtitles_settings.search.saveAs = ConfigSelection(default='version', choices=[('default', _("Default")), ('version', _("Release")), ('video', _("Video filename"))])
    subtitles_settings.search.saveTo = ConfigSelection(default='custom', choices=[('custom', _('User defined')), ('video', _('Next to video'))])
    subtitles_settings.search.addLangToSubsFilename = ConfigYesNo(default=False)
    subtitles_settings.search.askForDownloadLocation = ConfigYesNo(default=False)
    subtitles_settings.search.askOverwriteExistingSubs = ConfigYesNo(default=True)
    subtitles_settings.search.openParamsDialogOnSearch = ConfigYesNo(default=True)
    subtitles_settings.search.showProvidersErrorMessage = ConfigYesNo(default=True)
    return subtitles_settings

class SubsSupportEmbedded(object):

    def __init__(self, embeddedSupport, preferEmbedded):
        self.embeddedSupport = embeddedSupport
        self.preferEmbedded = preferEmbedded
        self.__subStyles = {}
        self.selected_subtitle = None
        self.subtitle_window = self.session.instantiateDialog(SubsEmbeddedScreen, self.subsSettings.embedded)
        self.subtitle_window.hide()
        if isinstance(self, InfoBarBase):
            self.__event_tracker = ServiceEventTracker(screen=self, eventmap=
                {
                    iPlayableService.evStart: self.__serviceChanged,
                    iPlayableService.evEnd: self.__serviceChanged,
                    # iPlayableService.evUpdatedInfo: self.__updatedInfo
                })
            self.onClose.append(self.exitEmbeddedSubs)

    def __isEmbeddedEnabled(self):
        return self.subtitle_window.shown

    embeddedEnabled = property(__isEmbeddedEnabled)

    def getCurrentServiceSubtitle(self):
        service = self.session.nav.getCurrentService()
        return service and service.subtitle()

    def __serviceChanged(self):
        if self.selected_subtitle:
            self.selected_subtitle = None
            self.subtitle_window.hide()

    def __updatedInfo(self):
        if not self.selected_subtitle:
            subtitle = self.getCurrentServiceSubtitle()
            cachedsubtitle = subtitle.getCachedSubtitle()
            if cachedsubtitle:
                self.enableSubtitle(cachedsubtitle)

    def enableSubtitle(self, selectedSubtitle):
        print '[SubsSupportEmbedded] enableSubtitle', selectedSubtitle
        subtitle = self.getCurrentServiceSubtitle()
        self.selected_subtitle = selectedSubtitle
        if subtitle and self.selected_subtitle:
            self.resetEmbeddedSubs()
            subtitle.enableSubtitles(self.subtitle_window.instance, self.selected_subtitle)
            self.subtitle_window.show()
            print '[SubsSupportEmbedded] enable embedded subtitles'
        else:
            print '[SubsSupportEmbedded] disable embedded subtitles'
            if subtitle:
                subtitle.disableSubtitles(self.subtitle_window.instance)
            self.subtitle_window.hide()

    def restartSubtitle(self):
        if self.selected_subtitle:
            print '[SubsSupportEmbedded] restart embedded subtitles'
            self.enableSubtitle(self.selected_subtitle)

    def resetEmbeddedSubs(self, reloadScreen=False):
        if QuickSubtitlesConfigMenu:
            return
        print '[SubsSupportEmbedded] updating  embedded screen'
        from enigma import eWidget, eSubtitleWidget
        scale = ((1, 1), (1, 1))
        embeddedSettings = self.subsSettings.embedded
        fontSize = embeddedSettings.fontSize.value
        fontType = embeddedSettings.fontType.value
        foregroundColor = "#" + embeddedSettings.color.value
        foregroundColor = parseColor(foregroundColor)
        borderColor = "#" + embeddedSettings.shadow.color.value
        borderColor = parseColor(borderColor)
        borderWidth = int(embeddedSettings.shadow.size.value)
        offset = "%s,%s" % (embeddedSettings.shadow.xOffset.value,
                          embeddedSettings.shadow.yOffset.value)
        shadowOffset = parsePosition(offset, scale)
        if fontType not in FONT:
            fontType = "Default"
            embeddedSettings.fontType.value = "Default"
            embeddedSettings.fontType.save()
        fontRegular = parseFont("%s;%s" % (FONT[fontType]['regular'], fontSize), scale)
        fontItalic = parseFont("%s;%s" % (FONT[fontType]['italic'], fontSize), scale)
        fontBold = parseFont("%s;%s" % (FONT[fontType]['bold'], fontSize), scale)
        self._loadEmbeddedStyle({"Subtitle_Regular":(fontRegular, 1, foregroundColor, borderColor, borderWidth, borderColor, shadowOffset),
                                "Subtitle_Italic":(fontItalic, 1, foregroundColor, borderColor, borderWidth, borderColor, shadowOffset),
                                "Subtitle_Bold":(fontBold, 1, foregroundColor, borderColor, borderWidth, borderColor, shadowOffset)})
        if reloadScreen:
            print '[SubsSupportEmbedded] reloading embedded screen'
            subtitle = self.getCurrentServiceSubtitle()
            if subtitle:
                subtitle.disableSubtitles(self.subtitle_window.instance)
            self.session.deleteDialog(self.subtitle_window)
            self.subtitle_window = None
            self.subtitle_window = self.session.instantiateDialog(SubsEmbeddedScreen, self.subsSettings.embedded)
            self.subtitle_window.hide()
            self.restartSubtitle()

    def _parseEmbeddedStyles(self, filename):
        if filename in self.__subStyles:
            return self.defaultStyles[filename]
        skin = parse_xml(filename).getroot()
        for c in skin.findall("subtitles"):
            scale = ((1, 1), (1, 1))
            substyles = {}
            for substyle in c.findall("sub"):
                get_attr = substyle.attrib.get
                font = parseFont(get_attr("font"), scale)
                col = get_attr("foregroundColor")
                if col:
                    foregroundColor = parseColor(col)
                    haveColor = 1
                else:
                    foregroundColor = gRGB(0xFFFFFF)
                    haveColor = 0
                col = get_attr("borderColor")
                if col:
                    borderColor = parseColor(col)
                else:
                    borderColor = gRGB(0)
                borderwidth = get_attr("borderWidth")
                if borderwidth is None:
                    # default: use a subtitle border
                    borderWidth = 3
                else:
                    borderWidth = int(borderwidth)
                col = get_attr("shadowColor")
                if col:
                        shadowColor = parseColor(col)
                else:
                        shadowColor = gRGB(0)
                col = get_attr("shadowOffset")
                if col:
                    shadowOffset = get_attr("shadowOffset")
                else:
                    shadowOffset = "-3,-3"
                shadowOffset = parsePosition(shadowOffset, scale)
                substyles[get_attr("name")] = (font, haveColor, foregroundColor, borderColor, borderWidth, shadowColor, shadowOffset)
                self.__subStyles[filename] = substyle
            return substyles

    def _loadEmbeddedStyle(self, substyles):
        from enigma import eWidget, eSubtitleWidget
        for faceName in substyles.keys():
            s = substyles[faceName]
            face = eSubtitleWidget.__dict__[faceName]
            font, haveColor, foregroundColor, borderColor, borderWidth, shadowColor, shadowOffset = s[0], s[1], s[2], s[3], s[4], s[5], s[6]
            try:
                eSubtitleWidget.setFontStyle(face, font , haveColor, foregroundColor, borderColor, borderWidth)
            except TypeError:
                eSubtitleWidget.setFontStyle(face, font, haveColor, foregroundColor, shadowColor, shadowOffset)

    def resetEmbeddedDefaults(self):
        userSkin = resolveFilename(SCOPE_SKIN, 'skin_user.xml')
        defaultSkin = resolveFilename(SCOPE_SKIN, 'skin_default.xml')
        skinSubtitles = resolveFilename(SCOPE_SKIN, 'skin_subtitles.xml')
        skinPaths = [userSkin, skinSubtitles, defaultSkin]
        for skinPath in skinPaths:
            if fileExists(skinPath):
                styles = self._parseEmbeddedStyles(skinPath)
                if styles:
                    print "[SubsEmbeddedSupport] reseting defaults from", skinPath
                    self._loadEmbeddedStyle(styles)
                    break

    def exitEmbeddedSubs(self):
        if self.subtitle_window is not None:
            self.session.deleteDialog(self.subtitle_window)
            self.subtitle_window = None
        if not QuickSubtitlesConfigMenu:
            self.resetEmbeddedDefaults()


class SubsSupport(SubsSupportEmbedded):
    """Client class for subtitles

        If this class is not subclass of InfoBarBase  you should  use public function of this class to
        to connect your media player (resume,pause,exit,after seeking, subtitles setup)
        functions with subtitles

    @param session: set active session
    @param subsPath: set path for subtitles to load
    @param defaultPath: set default path when choosing external subtitles
    @param forceDefaultPath: always use default path when choosing external subtitles
    @param autoLoad: tries to auto load  subtitles according to name of played file
    @param embeddedSupport: added support for embedded subtitles
    """

    def __init__(self, session=None, subsPath=None, defaultPath=None, forceDefaultPath=False, autoLoad=True,
                 showGUIInfoMessages=True, embeddedSupport=False, preferEmbedded=False, searchSupport=False, configEntry=None):
        if session is not None:
            self.session = session
        self.searchSupport = searchSupport
        self.subsSettings = initSubsSettings(configEntry)
        SubsSupportEmbedded.__init__(self, embeddedSupport, preferEmbedded)
        self.__subsScreen = self.session.instantiateDialog(self._getSubsScreenCls(), self.subsSettings.external)
        self.__subsScreen.hide()
        self.__subsEngine = self._getSubsEngineCls()(self.session, self.subsSettings.engine, self.__subsScreen)
        self.__subsLoader = SubsLoader(PARSERS, ALL_LANGUAGES_ENCODINGS + ENCODINGS[self.subsSettings.encodingsGroup.getValue()])
        rowParsing = not self.subsSettings.engine.type.getValue() in ['standard', 'standard_new']
        self.__subsLoader.set_row_parsing(rowParsing)
        self.__loaded = False
        self.__working = False
        self.__firstStart = True
        self.__autoLoad = autoLoad
        self.__subsPath = None
        self.__subsDir = None
        self.__subsEnc = None
        self.__playerDelay = 0
        self.__subsDelay = 0
        self.__startDelay = int(self.subsSettings.engine.expert.startDelay.value)
        self.__defaultPath = None
        self.__isServiceSet = False
        self.__subclassOfInfobarBase = isinstance(self, InfoBarBase)
        self.__forceDefaultPath = forceDefaultPath
        self.__showGUIInfoMessages = showGUIInfoMessages
        self.__checkTimer = eTimer()
        self.__starTimer = eTimer()
        try:
                      self.__starTimer_conn = self.__starTimer.timeout.connect(self.__updateSubs)
        except AttributeError:
                      self.__starTimer.callback.append(self.__updateSubs)
#        self.__starTimer.callback.append(self.__updateSubs)
        try:
            from Screens.InfoBar import InfoBar
            InfoBar.instance.subtitle_window.hide()
        except Exception:
            pass

        if self.__subclassOfInfobarBase:
            self.__event_tracker = ServiceEventTracker(screen=self, eventmap=
            {
                iPlayableService.evStart: self.__serviceStarted,
                iPlayableService.evEnd: self.__serviceStopped,
                iPlayableService.evSeekableStatusChanged: self.__seekableStatusChanged,
            })
            self["SubsActions"] = HelpableActionMap(self, "SubtitlesActions",
                {
                "subtitles": (self.subsMenu, _("show subtitles menu")),
                } , -5)

            self.onClose.append(self.exitSubs)

        if defaultPath is not None and os.path.isdir(defaultPath):
            self.__defaultPath = defaultPath
            self.__subsDir = defaultPath

        if subsPath is not None and self.__autoLoad:
            self.loadSubs(subsPath)

    def loadSubs(self, subsPath, newService=True):
        """loads subtitles from subsPath
        @param subsPath: path to subtitles (http url supported)
        @param newService: set False if service remains the same
        @return: True if subtitles was successfully loaded
        @return: False if subtitles wasnt successfully loaded
        """
        self.__working = True
        self.__subsPath = None
        if self.__defaultPath is not None:
            self.__subsDir = self.__defaultPath
        else:
            self.__subsDir = None

        if subsPath is not None:
            if not subsPath.startswith('http'):
                if self.__defaultPath is not None and self.__forceDefaultPath:
                    self.__subsDir = self.__defaultPath
                else:
                    if os.path.isdir(os.path.dirname(subsPath)):
                        self.__subsDir = os.path.dirname(subsPath)
                    else:
                        self.__subsDir = self.__defaultPath
                if not os.path.isfile(subsPath):
                    print '[Subtitles] trying to load not existing path:', subsPath
                    subsPath = None

            if subsPath is not None:
                subsList, self.__subsEnc = self.__processSubs(subsPath, self.__subsEnc)
                if subsList is not None:
                    self.__subsPath = subsPath
                    if newService:
                        self.__subsEngine.reset()
                    self.__subsEngine.pause()
                    self.__subsEngine.setPlayerDelay(self.__playerDelay)
                    self.__subsEngine.setSubsList(subsList)
                    self.__loaded = True
                    self.__working = False
                    return True
                else:
                    self.__subsEnc = None
                    self.__subsPath = None
        self.__working = False
        return False


    def startSubs(self, time):
        """If subtitles are loaded then start to play them after time set in ms"""
        def wrapped():
            self.__startTimer.start(time, True)

        if self.__working or self.__loaded:
            self.__afterWork(wrapped)

    def isSubsLoaded(self):
        return self.__loaded

    def getSubsFileFromSref(self):
        ref = self.session.nav.getCurrentlyPlayingServiceReference()
        if os.path.isdir(os.path.dirname(ref.getPath())):
            self.__subsDir = os.path.dirname(ref.getPath())
            for parser in PARSERS:
                for ext in parser.parsing:
                    subsPath = os.path.splitext(ref.getPath())[0] + ext
                    if os.path.isfile(subsPath):
                        return subsPath
        return None

    def resumeSubs(self):
        if self.__loaded:
            print '[Subtitles] resuming subtitles'
            self.showSubsDialog()
            self.__subsEngine.resume()

    def pauseSubs(self):
        if self.__loaded:
            print '[Subtitles] pausing subtitles'
            self.__subsEngine.pause()

    def playAfterSeek(self):
        if self.__loaded:
            self.showSubsDialog()
            self.__subsEngine.sync()

    def showSubsDialog(self):
        if self.__loaded:
            print '[Subtitles] show dialog'
            self.__subsScreen.show()

    def hideSubsDialog(self):
        if self.__loaded:
            print '[Subtitles] hide dialog'
            if self.__subsScreen:
                self.__subsScreen.hide()

    def setPlayerDelay(self, delayInMs):
        self.__playerDelay = delayInMs

    def setSubsDelay(self, delayInMs):
        if self.__loaded:
            self.__subsDelay = delayInMs
            self.__subsEngine.setSubsDelay(self.__subsDelay)

    def getSubsDelay(self):
        if self.__loaded:
            return self.__subsDelay
        return None
    
    def getSubsPath(self):
        return self.__subsPath

    def subsMenu(self):
        if not self.__working and not (self.__subclassOfInfobarBase and not self.__isServiceSet):
            self.__alreadyPausedVideo = False
            if self.subsSettings.pauseVideoOnSubtitlesMenu.value:
                print '[SubsSupport] stopVideoOnSubtitlesMenu: True'
                if isinstance(self, InfoBarSeek):
                    if self.seekstate == InfoBarSeek.SEEK_STATE_PLAY:
                        print '[SubsSupport] pausing video'
                        self.setSeekState(InfoBarSeek.SEEK_STATE_PAUSE)
                    else:
                        print '[SubsSupport] video is already paused'
                        self.__alreadyPausedVideo = True
                else:
                    print '[SubsSupport] not subclass of InfobarSeek'
            self.session.openWithCallback(self.__subsMenuCB, SubsMenu, self,
                                          self.__subsPath, self.__subsDir, self.__subsEnc, self.embeddedSupport, self.embeddedEnabled, self.searchSupport)

    def resetSubs(self, resetEnc=True, resetEngine=True, newSubsScreen=False, newService=True):
        """
        Resets subtitle state -> stops engine, reload encodings, reset paths..
        @param resetEnc : start trying encodings from beginning of  current encodings-group list
        @param resetEngine: clean active subtitle, subtitle list, reset engine vars
        @param newSubsSCreen: recreates subtitles screen
        @param newService: set to True if new servicereference is in use
        """
        # start trying encodings from beginning of encodings_group list
        if resetEnc:
            self.__subsEnc = None
        self.__subsLoader.change_encodings(ALL_LANGUAGES_ENCODINGS + ENCODINGS[self.subsSettings.encodingsGroup.getValue()])
        self.__subsEngine.pause()
        # stop subtitles, clean active subtitle,  subtitles list, reset delay
        # if new service -> remove service
        if resetEngine:
            self.__subsDelay = 0
            self.__subsEngine.reset()
            if newService:
                self.__firstStart = False
        # hide subtitles, reload screen with new settings
        # if  newSubsScreen, remove current subscreen and create new one
        self.__resetSubsScreen(newSubsScreen)
        self.__subsPath = None
        self.__loaded = False

    def __resetSubsScreen(self, newSubsScreen=False):
        self.__subsScreen.hide()
        if newSubsScreen:
            self.__subsEngine.setRenderer(None)
            self.session.deleteDialog(self.__subsScreen)
            self.__subsScreen = self.session.instantiateDialog(self._getSubsScreenCls(), self.subsSettings.external)
            self.__subsEngine.setRenderer(self.__subsScreen)
        else:
            self.__subsScreen.reloadSettings()
        self.__subsScreen.show()


    def exitSubs(self):
        """This method should be called at the end of usage of this class"""
        self.hideSubsDialog()

        if self.__subsEngine:
            self.__subsEngine.exit()
            self.__subsEngine = None

        if self.__subsScreen:
            self.session.deleteDialog(self.__subsScreen)
            self.__subsScreen = None

        self.__starTimer.stop()
        self.__starTimer = None

        self.__checkTimer.stop()
        self.__checkTimer = None

        print '[SubsSupport] closing subtitleDisplay'


    def __subsMenuCB(self, subsPath, subsEmbedded, settingsChanged, changeEncoding,
                     changedEncodingGroup, changedShadowType, changedEngine, reloadEmbeddedScreen, turnOff, forceReload=False):
        if self.embeddedEnabled and self.embeddedSupport and not subsEmbedded and not turnOff and not subsPath:
            print "embedded settings changed"
            self.resetEmbeddedSubs(reloadEmbeddedScreen)
        elif turnOff:
            print '[SubsSupport] turn off'
            if self.embeddedSupport and self.embeddedEnabled:
                self.enableSubtitle(None)
            if self.__loaded:
                self.resetSubs(newService=False)
        elif self.embeddedSupport and subsEmbedded:
            print '[SubsSupport] loading embedded subtitles'
            if self.__loaded:
                self.resetSubs()
            self.__subsScreen.hide()
            self.enableSubtitle(subsEmbedded)
        elif subsPath is not None:
            if self.embeddedEnabled:
                self.enableSubtitle(None)
            newScreen = (changedEngine or changedShadowType)
            self.__subsScreen.show()
            if self.__subsPath == subsPath:
                if not settingsChanged and not ((changeEncoding or changedEncodingGroup) or newScreen or forceReload):
                    print '[SubsSupport] no changes made'
                elif settingsChanged and not (newScreen or changedEncodingGroup or forceReload):
                    print '[SubSupport] reloading SubScreen'
                    self.__subsEngine.pause()
                    self.__resetSubsScreen()
                    self.__subsEngine.resume()
                else:
                    self.__subsEngine.pause()
                    if changedEncodingGroup or ((changedEngine or changedShadowType) and not changeEncoding) or forceReload:
                        self.__subsEnc = None
                    if changedEncodingGroup:
                        self.__subsLoader.change_encodings(ALL_LANGUAGES_ENCODINGS + ENCODINGS[self.subsSettings.encodingsGroup.getValue()])
                    if changedEngine:
                        rowParsing = not self.subsSettings.engine.type.getValue() in ['standard', 'standard_new']
                        self.__subsLoader.set_row_parsing(rowParsing)
                    if newScreen:
                        self.__resetSubsScreen(newSubsScreen=True)
                    self.__subsEngine.reset(position=False)
                    if self.loadSubs(subsPath, newService=False):
                        self.__subsEngine.refresh()
            else:
                self.pauseSubs()
                self.__subsEnc = None
                if changedEngine:
                    rowParsing = not self.subsSettings.engine.type.getValue() in ['standard', 'standard_new']
                    self.__subsLoader.set_row_parsing(rowParsing)
                if changedEncodingGroup:
                        self.__subsLoader.change_encodings(ALL_LANGUAGES_ENCODINGS + ENCODINGS[self.subsSettings.encodingsGroup.getValue()])
                if newScreen:
                    self.__resetSubsScreen(newSubsScreen=True)
                self.__subsEngine.reset()
                if self.__loaded:
                    if self.loadSubs(subsPath, newService=False):
                        self.__subsEngine.refresh()
                else:
                    if self.loadSubs(subsPath, newService=False):
                        self.__subsEngine.resume()
        if not self.__alreadyPausedVideo and isinstance(self, InfoBarSeek):
            print '[SubsSupport] unpausing video'
            del self.__alreadyPausedVideo
            self.setSeekState(InfoBarSeek.SEEK_STATE_PLAY)


    def __processSubs(self, subsPath, subsEnc):
        showMessages = self.__showGUIInfoMessages and not (self.__firstStart and self.__subclassOfInfobarBase)
        try:
            return self.__subsLoader.load(subsPath, subsEnc, self._getFps())
        except LoadError:
            if showMessages:
                warningMessage(self.session, _("Cannot load subtitles. Invalid path"))
            return None, None
        except DecodeError:
            if showMessages:
                warningMessage(self.session, _("Cannot decode subtitles. Try another encoding group"))
            return None, None
        except ParserNotFoundError:
            if showMessages:
                warningMessage(self.session, _("Cannot parse subtitles. Not supported subtitles format"))
            return None, None
        except ParseError:
            if showMessages:
                warningMessage(self.session, _("Cannot parse subtitles. Invalid subtitles format"))
            return None, None
        finally:
            self.__firstStart = False

    def __updateSubs(self):
        if self.__loaded:
            self.resumeSubs()
            return

        subsPath = self.getSubsFileFromSref()
        if subsPath is not None:
            if self.loadSubs(subsPath):
                self.resumeSubs()
        self.__working = False

    def _getSubsScreenCls(self):
        if self.subsSettings.engine.type.value == 'standard':
            return SubsScreen
        elif self.subsSettings.engine.type.value == 'standard_new':
            return SubsScreenNew
        else:
            return SubsRowScreen

    def _getFps(self):
        from enigma import iServiceInformation
        service = self.session.nav.getCurrentService()
        info = service and service.info()
        if not info:
            return None
        fps = info.getInfo(iServiceInformation.sFrameRate)
        if fps > 0:
            return float("%6.3f" % (fps / 1000.))
        return None

    def _getSubsEngineCls(self):
        return SubsEngine

    def __afterWork(self, fnc):
        def checkWorking():
            if self.__working:
                print 'check working..'
                self.__checkTimer.start(200, True)
            else:
                self.__checkTimer.stop()
                fnc()

        self.__checkTimer.stop()
        self.__starTimer.stop()

        if self.__working:
            del self.__checkTimer.callback[:]
            self.__checkTimer.callback.append(checkWorking)
            self.__checkTimer.start(200, True)
        else:
            fnc()


############ Methods triggered by videoEvents when SubsSupport is subclass of Screen ################
    def __serviceStarted(self):
        print '[SubsSupport] Service Started'
        def startSubs():
            self.__starTimer.start(self.__startDelay, True)

        self.__isServiceSet = True
        # subtitles are loading or already loaded
        if self.__working or self.__loaded:
            self.__afterWork(startSubs)
        else:
            self.resetSubs(True)
            if self.__subsPath is None and self.__autoLoad:
                startSubs()

    def __serviceStopped(self):
        self.resetSubs(True)
        self.__isServiceSet = False

    def __seekableStatusChanged(self):
        if not hasattr(self, 'seekstate'):
            return
        if self.seekstate == self.SEEK_STATE_PLAY:
            self.pauseSubs()
        elif self.seekstate == self.SEEK_STATE_PAUSE:
            self.resumeSubs()
        elif self.seekstate == self.SEEK_STATE_EOF:
            self.resetSubs(True)

########### Methods which extends InfobarSeek seek methods

    def doSeekRelative(self, pts):
        if self.__loaded:
            # self.__subsEngine.preSeek(pts)
            super(SubsSupport, self).doSeekRelative(pts)
            self.playAfterSeek()
        else:
            super(SubsSupport, self).doSeekRelative(pts)
    def doSeek(self, pts):
        if self.__loaded:
            super(SubsSupport, self).doSeek(pts)
            self.playAfterSeek()
        else:
            super(SubsSupport, self).doSeek(pts)

############################################################

class SubsEmbeddedScreen(Screen):

    """
    Pli defaults
    <fonts>
        <font filename="nmsbd.ttf" name="Subs" scale="100" />
    </fonts>
    <subtitles>
        <sub name="Subtitle_TTX" font="Subs;34" borderColor="#000000" borderWidth="3" />
        <sub name="Subtitle_Regular" font="Subs;34" foregroundColor="#ffffff" borderColor="#000000" borderWidth="3" />
        <sub name="Subtitle_Bold" font="Subs;34" foregroundColor="#ffffff" borderColor="#000000" borderWidth="3" />
        <sub name="Subtitle_Italic" font="Subs;34" foregroundColor="#ffffff" borderColor="#000000" borderWidth="3" />
    </subtitles>
    """

    def __init__(self, session, embeddedSettings):
        desktop = getDesktop(0)
        size = desktop.size()
        vSizeOrig = size.height()
        hSizeOrig = size.width()
        if QuickSubtitlesConfigMenu:
            vPosition = 0
            vSize = vSizeOrig
        else:
            vPositionPercent = int(embeddedSettings.position.value)
            fontSize = int(getEmbeddedFontSizeCfg(embeddedSettings.fontSize).value)
            vSize = fontSize * 4 + 10
            vPosition = int(vPositionPercent * float((vSizeOrig - vSize) / 100))
            vPosition = vPosition if vPosition > 0 else 0
            vSize = vSizeOrig - vPosition
        self.skin = """<screen position="0,%s" size="%s,%s" zPosition="-1" backgroundColor="transparent" flags="wfNoBorder" />""" % (vPosition, hSizeOrig, vSize)
        Screen.__init__(self, session)


class SubtitlesWidget(GUIComponent):
    STATE_NO_BACKGROUND, STATE_BACKGROUND = range(2)

    def __init__(self, boundDynamic=True, boundXOffset=10, boundYOffset=10, boundSize=None, fontSize=25, positionPercent=94):
        GUIComponent.__init__(self)
        self.state = self.STATE_BACKGROUND
        self.boundDynamic = boundDynamic
        self.boundXOffset = boundXOffset
        self.boundYOffset = boundYOffset
        self.font = (gFont("Regular", fontSize), fontSize)
        self.positionPercent = positionPercent
        desktopSize = getDesktop(0).size()
        self.desktopSize = (desktopSize.width(), desktopSize.height())
        self.boundSize = boundSize or (self.desktopSize[0], self.calcWidgetHeight())

    def GUIcreate(self, parent):
        self.instance = eLabel(parent)
        self.instance2 = eLabel(parent)
        self.postWidgetCreate()

    def GUIdelete(self):
        self.preWidgetRemove()
        self.instance = None
        self.instance2 = None

    def postWidgetCreate(self):
        self.instance2.hide()
        self.update()

    def preWidgetRemove(self):
        pass

    def calcWidgetYPosition(self):
        return int((self.desktopSize[1] - self.calcWidgetHeight() - self.boundYOffset) / float(100) * self.positionPercent)

    def calcWidgetHeight(self):
        return int(4 * self.font[1] + 15)

    def update(self):
        ds = self.desktopSize
        bs = self.boundSize = (self.desktopSize[0], self.calcWidgetHeight())
        self.instance2.resize(eSize(int(ds[0]), int(bs[1])))
        self.instance2.move(ePoint(int(0), int(self.calcWidgetYPosition())))
        self.instance2.setHAlign(self.instance2.alignCenter)
        self.instance2.setVAlign(self.instance2.alignCenter)
        self.instance2.setFont(self.font[0])
        self.instance2.setTransparent(True)
        if not self.boundDynamic:
            self.instance.resize(eSize(int(bs[0]), int(bs[1])))
            self.instance.move(ePoint(int(ds[0] / 2 - bs[0] / 2), int(self.calcWidgetYPosition())))
        self.instance.setFont(self.font[0])
        self.instance.setHAlign(self.instance.alignCenter)
        self.instance.setVAlign(self.instance.alignCenter)

    def setText(self, text):
        if self.instance and self.instance2:
            if self.state == self.STATE_NO_BACKGROUND:
                self.instance.hide()
                self.instance2.setText(text)
                self.instance2.show()
            elif self.state == self.STATE_BACKGROUND:
                self.instance2.hide()
                if not text:
                    self.instance.hide()
                    return
                if self.boundDynamic:
                    # hack so empty spaces are part of calculateSize calculation
                    self.instance2.setText(text.replace(' ','.'))
                    ds = self.desktopSize
                    bs = self.boundSize
                    ws = self.instance2.calculateSize()
                    ws = (ws.width() + self.boundXOffset * 2, ws.height() + self.boundYOffset * 2)
                    wp = self.instance2.position()
                    wp = (wp.x(), wp.y())
                    wpy = wp[1] + (bs[1] - ws[1]) / 2
                    wpx = ds[0] / 2 - ws[0] / 2
                    self.instance.resize(eSize(int(ws[0]), int(ws[1])))
                    self.instance.move(ePoint(int(wpx), int(wpy)))
                else:
                    bs = self.boundSize
                    ds = self.desktopSize
                    self.instance.resize(eSize(int(bs[0]), int(bs[1])))
                    self.instance.move(ePoint(int(ds[0] / 2 - bs[0] / 2), int(self.calcWidgetYPosition())))
                self.instance.setHAlign(self.instance.alignCenter)
                self.instance.setVAlign(self.instance.alignCenter)
                self.instance.setText(text)
                self.instance.show()

    def setPosition(self, percent):
        self.positionPercent = percent
        self.update()

    def setBoundDynamic(self, value):
        self.boundDynamic = value
        self.update()

    def setBoundOffset(self, offsetX, offsetY):
        self.boundXOffset = int(offsetX)
        self.boundYOffset = int(offsetY)
        self.update()

    def setForegroundColor(self, color):
        self.instance.setForegroundColor(parseColor(color))
        self.instance2.setForegroundColor(parseColor(color))

    def setBackgroundColor(self, color):
        if color[1:3] == "ff":
            self.state = self.STATE_NO_BACKGROUND
        else:
            self.state = self.STATE_BACKGROUND
            self.instance.setBackgroundColor(parseColor(color))

    def setBorderColor(self, color):
        self.instance.setBorderColor(parseColor(color))
        self.instance2.setBorderColor(parseColor(color))

    def setBorderWidth(self, width):
        self.instance.setBorderWidth(int(width))
        self.instance2.setBorderWidth(int(width))

    def setShadowColor(self, color):
        self.instance.setShadowColor(parseColor(color))
        self.instance2.setShadowColor(parseColor(color))

    def setShadowOffset(self, offset, scale):
        self.instance.setShadowOffset(parsePosition(offset, scale))
        self.instance2.setShadowOffset(parsePosition(offset, scale))

    def setFont(self, font):
        if self.font[1] == font[1]:
            self.font = font
            self.instance.setFont(font[0])
            self.instance2.setFont(font[0])
        else:
            self.font = font
            self.update()

class SubsScreenNew(Screen):
    def __init__(self, session, externalSettings):
        desktop = getDesktop(0)
        size = desktop.size()
        self.screenWidth = size.width()
        self.screenHeight = size.height()
        self.subShown = False
        self.__shadowType = 'border'
        self.__eLabelHasBorderParams = False
        self.externalSettings = externalSettings
        # eLabel on older versions of e2 doesnt have border option
        fontSize = int(externalSettings.fontSize.getValue())
        fontType = externalSettings.fontType.getValue()
        if fontType not in FONT:
            fontType = "Default"
            externalSettings.fontType.setValue("Default")
            externalSettings.fontType.save()

        self.font = {"regular":(gFont(FONT[fontType]['regular'], fontSize), fontSize),
                     "italic":(gFont(FONT[fontType]['italic'], fontSize), fontSize),
                     "bold":(gFont(FONT[fontType]['bold'], fontSize), fontSize)}

        self.selectedFont = "regular"
        color = externalSettings.color.getValue()
        self.currentColor = color

        self.skin = """
            <screen position="0,0" size="%s,%s" zPosition="-1" backgroundColor="transparent" flags="wfNoBorder">
                    <widget name="subtitles" />
            </screen>""" % (str(self.screenWidth), str(self.screenHeight))

        Screen.__init__(self, session)
        self.stand_alone = True
        print 'initializing subtitle display'

        self["subtitles"] = SubtitlesWidget()
        self.onLayoutFinish.append(self.__checkElabelCaps)
        self.onLayoutFinish.append(self.reloadSettings)

    def __checkElabelCaps(self):
        if hasattr(self["subtitles"].instance, 'setBorderWidth') and hasattr(self["subtitles"].instance, 'setBorderColor'):
            self.__eLabelHasBorderParams = True
        elif self.__shadowType == 'border':
            self.__shadowType = 'offset'

    def setShadowType(self, type):
        if type == 'border' and self.__eLabelHasBorderParams:
            self.__shadowType = 'border'
        else:
            self.__shadowType = 'offset'

    def setShadow(self, type, color, size=None, xOffset=None, yOffset=None):
        self.setShadowType(type)
        if self.__shadowType == 'border':
            self["subtitles"].setBorderColor("#" + color)
        elif self.__shadowType == 'offset':
            self["subtitles"].setShadowColor("#" + color)
        if self.__shadowType == 'border' and size is not None:
            self["subtitles"].setBorderWidth(size)
        elif self.__shadowType == 'offset' and (xOffset is not None and yOffset is not None):
            self["subtitles"].setShadowOffset(str(-xOffset) + ',' + str(-yOffset), self.scale)

    def setBackground(self, type, alpha, color, xOffset=None, yOffset=None):
        if type == 'dynamic':
            self["subtitles"].setBoundDynamic(True)
            self["subtitles"].setBoundOffset(xOffset, yOffset)
        else:
            self["subtitles"].setBoundDynamic(False)
        color = "#" + alpha + color
        self["subtitles"].setBackgroundColor(color)

    def setColor(self, color):
        self.currentColor = color
        color = "#" + color
        self["subtitles"].setForegroundColor(color)

    def setPosition(self, position):
        self["subtitles"].setPosition(position)

    def setFonts(self, font):
        self.font = font
        self['subtitles'].setFont(self.font['regular'])

    def reloadSettings(self):
        color = self.externalSettings.color.getValue()
        alpha = self.externalSettings.alpha.getValue()
        shadowType = self.externalSettings.shadow.type.getValue()
        shadowColor = self.externalSettings.shadow.color.getValue()
        shadowSize = int(self.externalSettings.shadow.size.getValue())
        shadowXOffset = int(self.externalSettings.shadow.xOffset.getValue())
        shadowYOffset = int(self.externalSettings.shadow.yOffset.getValue())
        shadowEnabled = int(self.externalSettings.shadow.enabled.getValue())
        if not shadowEnabled:
            shadowXOffset = shadowYOffset = shadowSize = 0
        backgroundType = self.externalSettings.background.type.getValue()
        backgroundAlpha = self.externalSettings.background.alpha.getValue()
        backgroundColor = self.externalSettings.background.color.getValue()
        backgroundXOffset = self.externalSettings.background.xOffset.getValue()
        backgroundYOffset = self.externalSettings.background.yOffset.getValue()
        backgroundEnabled = self.externalSettings.background.enabled.getValue()
        if not backgroundEnabled:
            backgroundAlpha = "ff"
            backgroundColor = "ffffff"
        fontSize = int(self.externalSettings.fontSize.getValue())
        position = int(self.externalSettings.position.getValue())

        self.setColor(alpha + color)
        self.setPosition(position)
        self.setShadow(shadowType, shadowColor, shadowSize, shadowXOffset, shadowYOffset)
        self.setBackground(backgroundType, backgroundAlpha, backgroundColor, backgroundXOffset, backgroundYOffset)
        self.setFonts({"regular":(gFont(FONT[self.externalSettings.fontType.getValue()]['regular'], fontSize), fontSize),
                       "italic":(gFont(FONT[self.externalSettings.fontType.getValue()]['italic'], fontSize), fontSize),
                       "bold":(gFont(FONT[self.externalSettings.fontType.getValue()]['bold'], fontSize), fontSize)})

    def setSubtitle(self, sub):
        self.subShown = True
        if sub['style'] != self.selectedFont:
            self.selectedFont = sub['style']
            self['subtitles'].setFont(self.font[sub['style']])
        if sub['color'] != 'default':
            self.setColor(sub['color'])
        elif self.currentColor != (self.externalSettings.alpha.getValue() + self.externalSettings.color.getValue()):
            self.setColor(self.externalSettings.alpha.getValue() + self.externalSettings.color.getValue())
        self["subtitles"].setText(sub['text'].encode('utf-8'))

    def hideSubtitle(self):
        if self.subShown:
            self.subShown = False
            self["subtitles"].setText("")


class SubsScreen(Screen):

    def __init__(self, session, externalSettings):
        desktop = getDesktop(0)
        size = desktop.size()
        self.screenWidth = size.width()
        self.screenHeight = size.height()
        self.subShown = False
        self.externalSettings = externalSettings
        # eLabel on older versions of e2 doesnt have border option
        self.__eLabelHasBorderParams = False
        self.__shadowType = 'border'
        fontSize = int(externalSettings.fontSize.getValue())
        fontType = externalSettings.fontType.getValue()
        if fontType not in FONT:
            fontType = "Default"
            externalSettings.fontType.setValue("Default")
            externalSettings.fontType.save()

        self.font = {"regular":gFont(FONT[fontType]['regular'], fontSize),
                     "italic":gFont(FONT[fontType]['italic'], fontSize),
                     "bold":gFont(FONT[fontType]['bold'], fontSize)}

        self.selectedFont = "regular"
        position = int(externalSettings.position.getValue())
        vSize = fontSize * 4 + 15  # 3 rows + reserve
        color = externalSettings.color.getValue()
        self.currentColor = color
        position = int(position * (float(self.screenHeight - vSize) / 100))

        self.skin = """
            <screen name="SubtitleDisplay" position="0,0" size="%s,%s" zPosition="-1" backgroundColor="transparent" flags="wfNoBorder">
                    <widget name="subtitles" position="0,%s" size="%s,%s" valign="center" halign="center" transparent="1"/>
            </screen>""" % (str(self.screenWidth), str(self.screenHeight), str(position), str(self.screenWidth), str(vSize))

        Screen.__init__(self, session)
        self.stand_alone = True
        print 'initializing subtitle display'

        self["subtitles"] = Label("")

        self.onLayoutFinish.append(self.__checkElabelCaps)
        self.onLayoutFinish.append(self.reloadSettings)

    def __checkElabelCaps(self):
        if hasattr(self["subtitles"].instance, 'setBorderWidth') and hasattr(self["subtitles"].instance, 'setBorderColor'):
            self.__eLabelHasBorderParams = True
        elif self.__shadowType == 'border':
            self.__shadowType = 'offset'

    def setShadowType(self, type):
        if type == 'border' and self.__eLabelHasBorderParams:
            self.__shadowType = 'border'
        else:
            self.__shadowType = 'offset'

    def setShadowWidth(self, width):
        self["subtitles"].instance.setBorderWidth(int(width))

    def setShadowColor(self, color, border=False):
        color = "#" + color
        if self.__shadowType == 'border':
            self["subtitles"].instance.setBorderColor(parseColor(color))
        else:
            self["subtitles"].instance.setShadowColor(parseColor(color))

    def setShadowOffset(self, offset):
        self["subtitles"].instance.setShadowOffset(parsePosition(offset, self.scale))

    def setShadow(self, type, color, size=None, xOffset=None, yOffset=None):
        self.setShadowType(type)
        self.setShadowColor(color)
        if self.__shadowType == 'border' and size is not None:
            self.setShadowWidth(size)
        elif self.__shadowType == 'offset' and (xOffset is not None and yOffset is not None):
            self.setShadowOffset(str(-xOffset) + ',' + str(-yOffset))

    def setColor(self, color):
        self.currentColor = color
        color = "#" + color
        self["subtitles"].instance.setForegroundColor(parseColor(color))

    def setPosition(self, position):
        self["subtitles"].instance.move(ePoint(0, position))

    def setFonts(self, font):
        self.font = font
        self['subtitles'].instance.setFont(self.font['regular'])

    def reloadSettings(self):
        color = self.externalSettings.color.getValue()
        alpha = self.externalSettings.alpha.getValue()
        shadowType = self.externalSettings.shadow.type.getValue()
        shadowColor = self.externalSettings.shadow.color.getValue()
        shadowSize = int(self.externalSettings.shadow.size.getValue())
        shadowXOffset = int(self.externalSettings.shadow.xOffset.getValue())
        shadowYOffset = int(self.externalSettings.shadow.yOffset.getValue())
        shadowEnabled = int(self.externalSettings.shadow.enabled.getValue())
        if not shadowEnabled:
            shadowXOffset = shadowYOffset = shadowSize = 0

        fontSize = int(self.externalSettings.fontSize.getValue())
        vSize = fontSize * 4 + 15
        position = int(self.externalSettings.position.getValue())
        position = int(position * (float(self.screenHeight - vSize) / 100))

        self.setColor(alpha + color)
        self.setPosition(position)
        self.setShadow(shadowType, shadowColor, shadowSize, shadowXOffset, shadowYOffset)
        self.setFonts({"regular":gFont(FONT[self.externalSettings.fontType.getValue()]['regular'], fontSize),
                       "italic":gFont(FONT[self.externalSettings.fontType.getValue()]['italic'], fontSize),
                       "bold":gFont(FONT[self.externalSettings.fontType.getValue()]['bold'], fontSize)})

    def setSubtitle(self, sub):
        self.subShown = True
        if sub['style'] != self.selectedFont:
            self.selectedFont = sub['style']
            self['subtitles'].instance.setFont(self.font[sub['style']])
        if sub['color'] != 'default':
            self.setColor(sub['color'])
        elif self.currentColor != (self.externalSettings.alpha.getValue() + self.externalSettings.color.getValue()):
            self.setColor(self.externalSettings.alpha.getValue() + self.externalSettings.color.getValue())
        self["subtitles"].setText(sub['text'].encode('utf-8'))

    def hideSubtitle(self):
        if self.subShown:
            self.subShown = False
            self["subtitles"].setText("")

class SubsRowScreen(Screen):
    def __init__(self, session, externalSettings):
        desktop = getDesktop(0)
        size = desktop.size()
        self.screenWidth = size.width()
        self.screenHeight = size.height()
        self.externalSettings = externalSettings
        self.subShown = False
        # eLabel on older versions of e2 doesnt have border option
        self.__eLabelHasBorderParams = False
        self.__shadowType = 'border'
        fontSize = int(externalSettings.fontSize.getValue())
        fontType = externalSettings.fontType.getValue()
        if fontType not in FONT:
            fontType = "Default"
            externalSettings.fontType.setValue("Default")
            externalSettings.fontType.save()

        self.font = {"regular":gFont(FONT[fontType]['regular'], fontSize),
                     "italic":gFont(FONT[fontType]['italic'], fontSize),
                     "bold":gFont(FONT[fontType]['bold'], fontSize)}

        position = int(externalSettings.position.getValue())
        vSize = fontSize * 4 + 15  # 3 rows + reserve
        position = int(position * (float(self.screenHeight - vSize) / 100))

        self.colorRow1 = 'default'
        self.colorRow2 = 'default'
        self.colorRow3 = 'default'
        self.colorRow4 = 'default'
        self.fontRow1 = 'regular'
        self.fontRow2 = 'regular'
        self.fontRow3 = 'regular'
        self.fontRow4 = 'regular'

        self.skin = """
            <screen name="SubtitleDisplay" position="0,0" size="%s,%s" zPosition="-1" backgroundColor="transparent" flags="wfNoBorder">
                    <widget name="subrow1" position="0,%s" size="%s,%s" valign="center" halign="center" transparent="1"/>
                    <widget name="subrow2" position="0,%s" size="%s,%s" valign="center" halign="center" transparent="1"/>
                    <widget name="subrow3" position="0,%s" size="%s,%s" valign="center" halign="center" transparent="1"/>
                    <widget name="subrow4" position="0,%s" size="%s,%s" valign="center" halign="center" transparent="1"/>
            </screen>""" % (str(self.screenWidth), str(self.screenHeight), str(position), str(self.screenWidth), str(vSize),
                            str(position), str(self.screenWidth), str(vSize), str(position), str(self.screenWidth), str(vSize), str(position), str(self.screenWidth), str(vSize))

        Screen.__init__(self, session)
        self.stand_alone = True
        print 'initializing subtitle display'

        self["subrow1"] = Label("")
        self["subrow2"] = Label("")
        self["subrow3"] = Label("")
        self["subrow4"] = Label("")

        self.onLayoutFinish.append(self.__checkElabelCaps)
        self.onLayoutFinish.append(self.reloadSettings)

    def __checkElabelCaps(self):
        if hasattr(self["subrow1"].instance, 'setBorderWidth') and hasattr(self["subrow1"].instance, 'setBorderColor'):
            self.__eLabelHasBorderParams = True
        elif self.__shadowType == 'border':
            self.__shadowType = 'offset'

    def setShadowType(self, type):
        if type == 'border' and self.__eLabelHasBorderParams:
            self.__shadowType = 'border'
        else:
            self.__shadowType = 'offset'

    def setShadowWidth(self, width):
        self["subrow1"].instance.setBorderWidth(int(width))
        self["subrow2"].instance.setBorderWidth(int(width))
        self["subrow3"].instance.setBorderWidth(int(width))
        self["subrow4"].instance.setBorderWidth(int(width))

    def setShadowColor(self, color, border=False):
        color = "#" + color
        if self.__shadowType == 'border':
            self["subrow1"].instance.setBorderColor(parseColor(color))
            self["subrow2"].instance.setBorderColor(parseColor(color))
            self["subrow3"].instance.setBorderColor(parseColor(color))
            self["subrow4"].instance.setBorderColor(parseColor(color))
        else:
            self["subrow1"].instance.setShadowColor(parseColor(color))
            self["subrow2"].instance.setShadowColor(parseColor(color))
            self["subrow3"].instance.setShadowColor(parseColor(color))
            self["subrow4"].instance.setShadowColor(parseColor(color))

    def setShadowOffset(self, offset):
        self["subrow1"].instance.setShadowOffset(parsePosition(offset, self.scale))
        self["subrow2"].instance.setShadowOffset(parsePosition(offset, self.scale))
        self["subrow3"].instance.setShadowOffset(parsePosition(offset, self.scale))
        self["subrow4"].instance.setShadowOffset(parsePosition(offset, self.scale))

    def setShadow(self, type, color, size=None, xOffset=None, yOffset=None):
        self.setShadowType(type)
        self.setShadowColor(color)
        if self.__shadowType == 'border' and size is not None:
            self.setShadowWidth(size)
        elif self.__shadowType == 'offset' and (xOffset is not None and yOffset is not None):
            self.setShadowOffset(str(-xOffset) + ',' + str(-yOffset))

    def setColor(self, color):
        color = "#" + color
        self["subrow1"].instance.setForegroundColor(parseColor(color))
        self["subrow2"].instance.setForegroundColor(parseColor(color))
        self["subrow3"].instance.setForegroundColor(parseColor(color))
        self["subrow4"].instance.setForegroundColor(parseColor(color))

    def setPosition(self, position):
        self["subrow1"].instance.move(ePoint(0, position))
        self["subrow2"].instance.move(ePoint(0, position))
        self["subrow3"].instance.move(ePoint(0, position))
        self["subrow4"].instance.move(ePoint(0, position))

    def setFonts(self, font):
        self.font = font
        self["subrow1"].instance.setFont(self.font['regular'])
        self["subrow2"].instance.setFont(self.font['regular'])
        self["subrow3"].instance.setFont(self.font['regular'])
        self["subrow4"].instance.setFont(self.font['regular'])

    def reloadSettings(self):
        color = self.externalSettings.color.getValue()
        alpha = self.externalSettings.alpha.getValue()
        shadowType = self.externalSettings.shadow.type.getValue()
        shadowColor = self.externalSettings.shadow.color.getValue()
        shadowSize = int(self.externalSettings.shadow.size.getValue())
        shadowXOffset = int(self.externalSettings.shadow.xOffset.getValue())
        shadowYOffset = int(self.externalSettings.shadow.yOffset.getValue())
        shadowEnabled = int(self.externalSettings.shadow.enabled.getValue())
        if not shadowEnabled:
            shadowXOffset = shadowYOffset = shadowSize = 0

        fontSize = int(self.externalSettings.fontSize.getValue())
        vSize = fontSize * 4 + 15
        position = int(self.externalSettings.position.getValue())
        position = int(position * (float(self.screenHeight - vSize) / 100))

        self.setColor(alpha + color)
        self.setPosition(position)
        self.setShadow(shadowType, shadowColor, shadowSize, shadowXOffset, shadowYOffset)
        self.setFonts({"regular":gFont(FONT[self.externalSettings.fontType.getValue()]['regular'], fontSize),
                       "italic":gFont(FONT[self.externalSettings.fontType.getValue()]['italic'], fontSize),
                       "bold":gFont(FONT[self.externalSettings.fontType.getValue()]['bold'], fontSize)})

    def setSubtitle(self, sub):
        self.subShown = True
        lenRows = len(sub['rows'])
        if lenRows == 1:
            self["subrow2"].setText("")
            self["subrow3"].setText("")
            self["subrow4"].setText("")
        elif lenRows == 2:
            self["subrow3"].setText("")
            self["subrow4"].setText("")
        elif lenRows == 3:
            self["subrow4"].setText("")
        for idx, row in enumerate(sub['rows']):
            self._setSubRow(row, idx + 1, lenRows)

    def _getSubRow(self, rowNum):
        return self.get('subrow' + str(rowNum))

    def _setSubRow(self, sub, rowNum, lenRows):
        if lenRows == 1:
            text = sub['text']
        elif lenRows == 2:
            if rowNum == 1:
                text = sub['text'] + u'\n'
            elif rowNum == 2:
                text = u'\n' + sub['text']
        elif lenRows == 3:
            if rowNum == 1:
                text = sub['text'] + u'\n\n'
            elif rowNum == 2:
                text = u'\n' + sub['text'] + u'\n'
            elif rowNum == 3:
                text = u'\n\n' + sub['text']
        elif lenRows == 4:
            if rowNum == 1:
                text = sub['text'] + u'\n\n\n'
            elif rowNum == 2:
                text = u'\n' + sub['text'] + u'\n\n'
            elif rowNum == 3:
                text = u'\n\n' + sub['text'] + u'\n'
            elif rowNum == 4:
                text = u'\n\n\n' + sub['text']
        else:
            return
        subRow = self._getSubRow(rowNum)

        if sub['style'] != 'regular':
            setattr(self, 'fontRow' + str(rowNum), sub['style'])
            subRow.instance.setFont(self.font[sub['style']])
        elif getattr(self, 'fontRow' + str(rowNum)) != 'regular':
            setattr(self, 'fontRow' + str(rowNum), sub['style'])
            subRow.instance.setFont(self.font['regular'])

        if sub['color'] != 'default':
            setattr(self, 'colorRow' + str(rowNum), sub['color'])
            color = "#" + sub['color']
            subRow.instance.setForegroundColor(parseColor(color))
        elif getattr(self, 'colorRow' + str(rowNum)) != 'default':
            setattr(self, 'colorRow' + str(rowNum), 'default')
            color = self.externalSettings.color.getValue()
            color = "#" + color
            subRow.instance.setForegroundColor(parseColor(color))
        subRow.setText(text.encode('utf-8'))

    def hideSubtitle(self):
        if self.subShown:
            self.subShown = False
            self["subrow1"].setText("")
            self["subrow2"].setText("")
            self["subrow3"].setText("")
            self["subrow4"].setText("")

class SubsEngine(object):
    def __init__(self, session, engineSettings, renderer):
        self.session = session
        self.engineSettings = engineSettings
        self.renderer = renderer
        self.subsList = None
        self.position = 0
        self.sub = None
        self.subsDelay = 0
        self.playerDelay = 0
        self.syncDelay = 300
        self.hideInterval = 200 * 90
        self.__seek = None
        self.__pts = None
        self.__ptsDelay = None
        self.__callbackPts = None
        self.preDoPlay = [self.updateSubPosition]
        self.refreshTimer = eTimer()
        try:
                      self.refreshTimer_conn = self.refreshTimer.timeout.connect(self.play)
        except AttributeError:
                      self.refreshTimer.callback.append(self.play)
#        self.refreshTimer.callback.append(self.play)
        self.refreshTimerDelay = 1000
        self.hideTimer = eTimer()
        try:
                      self.hideTimer_conn = self.hideTimer.timeout.connect(self.checkHideSub)
        except AttributeError:
                      self.hideTimer.callback.append(self.checkHideSub)
        
#        self.hideTimer.callback.append(self.checkHideSub)
        try:
                      self.hideTimer_conn = self.hideTimer.timeout.connect(self.incSubPosition)
        except AttributeError:
                      self.hideTimer.callback.append(self.incSubPosition)

#        self.hideTimer.callback.append(self.incSubPosition)
        try:
                      self.hideTimer_conn = self.hideTimer.timeout.connect(self.doPlay)
        except AttributeError:
                      self.hideTimer.callback.append(self.doPlay)
        
#        self.hideTimer.callback.append(self.doPlay)
        self.getPlayPtsTimer = eTimer()
        try:
                      self.getPlayPtsTimer_conn = self.getPlayPtsTimer.timeout.connect(self.getPts)
        except AttributeError:
                      self.getPlayPtsTimer.callback.append(self.getPts)

#        self.getPlayPtsTimer.callback.append(self.getPts)
        try:
                      self.getPlayPtsTimer_conn = self.getPlayPtsTimer.timeout.connect(self.validPts)
        except AttributeError:
                      self.getPlayPtsTimer.callback.append(self.validPts)

#        self.getPlayPtsTimer.callback.append(self.validPts)
        try:
                      self.getPlayPtsTimer_conn = self.getPlayPtsTimer.timeout.connect(self.callbackPts)
        except AttributeError:
                      self.getPlayPtsTimer.callback.append(self.callbackPts)
        
#        self.getPlayPtsTimer.callback.append(self.callbackPts)
        self.getPlayPtsTimerDelay = 200
        self.resume = self.play
        self.addNotifiers()

    def addNotifiers(self):
        def hideInterval(configElement):
            self.hideInterval = int(configElement.value) * 90
        def playerDelay(configElement):
                self.playerDelay = int(configElement.value) * 90
        def syncDelay(configElement):
            self.syncDelay = int(configElement.value)
        def getPlayPtsTimerDelay(configElement):
            self.getPlayPtsTimerDelay = int(configElement.value)
        def refreshTimerDelay(configElement):
            self.refreshTimerDelay = int(configElement.value)

        self.engineSettings.expert.hideDelay.addNotifier(hideInterval)
        self.engineSettings.expert.playerDelay.addNotifier(playerDelay)
        self.engineSettings.expert.syncDelay.addNotifier(syncDelay)
        self.engineSettings.expert.ptsDelayCheck.addNotifier(getPlayPtsTimerDelay)
        self.engineSettings.expert.refreshDelay.addNotifier(refreshTimerDelay)

    def removeNotifiers(self):
        del self.engineSettings.expert.hideDelay.notifiers[:]
        del self.engineSettings.expert.playerDelay.notifiers[:]
        del self.engineSettings.expert.syncDelay.notifiers[:]
        del self.engineSettings.expert.ptsDelayCheck.notifiers[:]
        del self.engineSettings.expert.refreshDelay.notifiers[:]

    def getPlayPts(self, callback, delay=None):
        self.getPlayPtsTimer.stop()
        self.__callbackPts = callback
        self.__ptsDelay = delay
        self.__pts = None
        if delay is None:
            delay = 1
        self.getPlayPtsTimer.start(delay, True)

    def getPts(self):
        try:
            if not self.__seek:
                service = self.session.nav.getCurrentService()
                self.__seek = service.seek()
        except Exception:
            return
        r = self.__seek.getPlayPosition()
        if r[0]:
            self.__pts = None
        else:
            self.__pts = long(r[1]) + self.playerDelay

    def validPts(self):
        pass

    def callbackPts(self):
        if self.__pts is not None:
            self.getPlayPtsTimer.stop()
            self.__callbackPts()
        else:
            delay = self.getPlayPtsTimerDelay
            if self.__ptsDelay is not None:
                delay = self.__ptsDelay
            self.getPlayPtsTimer.start(delay)

    def setSubsList(self, subslist):
        self.subsList = subslist

    def setRenderer(self, renderer):
        self.renderer = renderer

    def setPlayerDelay(self, playerDelay):
        self.pause()
        self.playerDelay = playerDelay * 90
        self.resume()

    def setSubsDelay(self, delay):
        self.pause()
        self.subsDelay = delay * 90
        self.resume()

    def reset(self, position=True):
        self.stopTimers()
        self.hideSub()
        if position:
            self.position = 0
        self.__seek = None
        self.__pts = None
        self.__callbackPts = None
        self.sub = None
        self.subsDelay = 0

    def refresh(self):
        self.stopTimers()
        self.hideSub()
        self.refreshTimer.start(self.refreshTimerDelay, True)

    def pause(self):
        self.stopTimers()
        self.hideSub()

    def play(self):
        self.stopTimers()
        self.hideSub()
        self.getPlayPts(self.prePlay)

    def sync(self):
        self._oldPts = None
        def checkPts():
            if self._oldPts is None:
                self._oldPts = self.__pts
                self.getPlayPts(checkPts, self.syncDelay)
            # video is frozen no progress made
            elif self._oldPts == self.__pts:
                self._oldPts = None
                self.getPlayPts(checkPts, self.syncDelay)
            # abnormal pts
            elif (self.__pts > self._oldPts + self.syncDelay * 90 + (200 * 90)) or (
                    self.__pts < self._oldPts + self.syncDelay * 90 - (200 * 90)):
                self._oldPts = None
                self.getPlayPts(checkPts, self.syncDelay)
                # normal playback
            else:
                del self._oldPts
                self.updateSubPosition()
                self.doPlay()
        self.stopTimers()
        self.hideSub()
        self.getPlayPts(checkPts, self.syncDelay)

    def prePlay(self):
        for f in self.preDoPlay:
            f()
        self.doPlay()

    def doPlay(self):
        if self.position == len(self.subsList):
            print  '[SubsEngine] reached end of subtitle list'
            self.position = len(self.subsList) - 1
            self.stopTimers()
        else:
            self.sub = self.subsList[self.position]
            self.getPlayPts(self.doWait)

    def doWait(self):
        subStartPts = self.sub['start'] + self.subsDelay
        if self.__pts < subStartPts:
            diffPts = subStartPts - self.__pts
            diffMs = diffPts / 90
            if diffMs > 50:
                self.getPlayPts(self.doWait, diffMs)
            else:
                print '[SubsEngine] sub shown sooner by %dms' % diffMs
                self.renderSub()
        else:
            if self.sub['end'] + self.subsDelay - self.__pts < 0:
                # print '[SubsEngine] %s < 0, skipping' % str(self.sub['end'] - self.__pts)
                self.getPlayPts(self.skipSubs, 100)
            else:
                print '[SubsEngine] sub shown later by %dms' % ((self.__pts - subStartPts) / 90)
                self.renderSub()

    def skipSubs(self):
        if self.position == len(self.subsList) - 1:
            self.incSubPosition()
        else:
            self.updateSubPosition()
        self.doPlay()

    def renderSub(self):
        duration = int(self.sub['duration'])
        self.renderer.setSubtitle(self.sub)
        self.hideTimer.start(duration, True)

    def checkHideSub(self):
        if self.subsList[-1] == self.sub:
            self.hideSub()
        elif self.subsList[self.position]['end'] + self.hideInterval < self.subsList[self.position + 1]['start']:
            self.hideSub()

    def hideSub(self):
        self.renderer.hideSubtitle()

    def incSubPosition(self):
        self.position += 1

    def updateSubPosition(self):
        playPts = self.__pts
        print '[SubsEngine] pre-update sub position:', self.position
        subStartPts = self.subsList[self.position]['start'] + self.subsDelay
        subEndPts = self.subsList[self.position]['end'] + self.subsDelay
        # seek backwards
        if subStartPts > playPts:
            subPrevEndPts = self.subsList[self.position -1]['end'] + self.subsDelay
            while self.position > 0 and subPrevEndPts > playPts:
                self.position -= 1
                subPrevEndPts = self.subsList[self.position - 1]['end'] + self.subsDelay
        # seek forward
        elif subStartPts < playPts:
            while self.position < len(self.subsList) - 1 and subStartPts < playPts:
                self.position += 1
                subStartPts = self.subsList[self.position]['start'] + self.subsDelay
        print '[SubsEngine] post-update sub position:', self.position

    def showDialog(self):
        self.renderer.show()

    def hideSubtitlesDialog(self):
        self.renderer.hide()

    def stopTimers(self):
        if self.refreshTimer is not None:
            self.refreshTimer.stop()
        if self.getPlayPtsTimer is not None:
            self.getPlayPtsTimer.stop()
        if self.hideTimer is not None:
            self.hideTimer.stop()

    def exit(self):
        self.hideTimer = None
        self.refreshTimer = None
        self.getPlayPtsTimer = None
        self.removeNotifiers()


class PanelList(MenuList):
    def __init__(self, list, height=30):
        MenuList.__init__(self, list, False, eListboxPythonMultiContent)
        self.l.setItemHeight(height)
        self.l.setFont(0, gFont("Regular", 20))
        self.l.setFont(1, gFont("Regular", 17))

def PanelListEntry(name, mode):
    res = [(name, mode)]
    res.append(MultiContentEntryText(pos=(5, 5), size=(330, 25), font=0, flags=RT_VALIGN_CENTER, text=name))
    return res

def PanelColorListEntry(name, value, colorName, colorValue, sizePanelX):
    res = [(name)]
    res.append(MultiContentEntryText(pos=(0, 5), size=(sizePanelX, 30), font=1, flags=RT_HALIGN_LEFT, text=name, color=colorName))
    res.append(MultiContentEntryText(pos=(0, 5), size=(sizePanelX, 30), font=1, flags=RT_HALIGN_RIGHT, text=value, color=colorValue))
    return res

class SubsMenu(Screen):
    skin = """
        <screen position="center,center" size="500,400" zPosition="1" >
            <widget name="title_label" position="0,5" size="500,35" valign="center" halign="center" font="Regular;25" transparent="1" foregroundColor="white" />
            <widget name="subfile_label" position="0,50" size="500,50" valign="center" halign="center" font="Regular;22" transparent="1" foregroundColor="#DAA520" />
            <widget name="subfile_list" position="center,100" size="300,30" transparent="1" />
            <eLabel position="5,135" size="490,1" backgroundColor="#999999" />
            <widget name="menu_list" position="0,140" size="500,235" transparent="1" scrollbarMode="showOnDemand" />
            <widget name="copyright" position="10,375" size="480,20" valign="center" halign="center" font="Regular;15" transparent="1" foregroundColor="white" />
        </screen>"""

    def __init__(self, session, infobar, subfile=None, subdir=None, encoding=None, embeddedSupport=False, embeddedEnabled=False, searchSupport=False):
        Screen.__init__(self, session)
        self.infobar = infobar
        self.subfile = subfile
        self.subdir = subdir
        self.encoding = encoding
        self.embeddedSupport = embeddedSupport
        self.embeddedEnabled = embeddedEnabled
        self.searchSupport = searchSupport
        self.embeddedSubtitle = None
        self.newSelection = False
        self.changeEncoding = False
        self.changedEncodingGroup = False
        self.changedShadowType = False
        self.changedEngine = False
        self.changedSettings = False
        self.reloadEmbeddedScreen = False
        self.turnOff = False
        self.forceReload = False

        self["title_label"] = Label(_("Currently choosed subtitles"))
        self["subfile_label"] = Label("")
        self["subfile_list"] = PanelList([], 25)
        self["menu_list"] = PanelList([], 28)
        self["copyright"] = Label("")
        # self["copyright"] = Label("created by %s <%s>"%(__author__,__email__))
        self["actions"] = ActionMap(["SetupActions", "DirectionActions"],
            {
                "ok": self.ok,
                "cancel": self.cancel,
            }, -2)

        self.onLayoutFinish.append(self.initTitle)
        self.onLayoutFinish.append(self.initGUI)
        self.onLayoutFinish.append(self.disableSelection)

    def disableSelection(self):
        self["subfile_list"].selectionEnabled(False)

    def initTitle(self):
        self.setTitle("SubsSupport %s" % __version__)

    def initGUI(self):
        self.initSubInfo()
        self.initMenu()

    def initSubInfo(self):
        subInfo = []
        if self.embeddedEnabled or self.embeddedSubtitle:
            self["subfile_label"].setText(_("Embedded Subtitles"))
            self["subfile_label"].instance.setForegroundColor(parseColor("#ffff00"))
        elif self.subfile is not None:
            self["subfile_label"].setText(toString(os.path.split(self.subfile)[1]))
            self["subfile_label"].instance.setForegroundColor(parseColor("#DAA520"))

            if self.newSelection:
                pass
                # subInfo.append(PanelColorListEntry(_("State:"),_("not loaded"), 0xDAA520, 0xffff00, 300))
            elif self.encoding and not self.newSelection:
                # subInfo.append(PanelColorListEntry(_("State:"),_("loaded"), 0xDAA520, 0x00ff00, 300))
                subInfo.append(PanelColorListEntry(_("Encoding:"), self.encoding, 0xDAA520, 0xffffff, 300))
            elif not self.encoding and not self.newSelection:
                # subInfo.append(PanelColorListEntry(_("State:"),_("not loaded"), 0xDAA520, 0xffff00, 300))
                subInfo.append(PanelColorListEntry(_("Encoding:"), _("cannot decode"), 0xDAA520, 0xffffff, 300))
        else:
            self["subfile_label"].setText(_("None"))
            self["subfile_label"].instance.setForegroundColor(parseColor("#DAA520"))
        self["subfile_list"].setList(subInfo)

    def initMenu(self):
        self.menu = [(_('Choose subtitles'), 'choose')]
        if not self.embeddedEnabled:
            if self.subfile is not None and not self.newSelection:
                self.menu.append((_('Change encoding'), 'encoding'))
            self.menu.append((_('Subtitles settings'), 'settings'))
        if self.embeddedEnabled and QuickSubtitlesConfigMenu:
            self.menu.append((_('Subtitles settings (embedded)'), 'settings_embedded_pli'))
        if self.embeddedEnabled and not QuickSubtitlesConfigMenu:
            self.menu.append((_('Subtitles settings (embedded)'), 'settings_embedded'))
        if self.subfile is not None or self.embeddedEnabled:
            self.menu.append((_('Turn off subtitles'), 'subsoff'))
        list = [PanelListEntry(x, y) for x, y in self.menu]
        self["menu_list"].setList(list)

    def ok(self):
        mode = self["menu_list"].getCurrent()[0][1]
        if mode == 'choose':
            self.session.openWithCallback(self.subsChooserCB, SubsChooser, self.infobar.subsSettings, self.subdir, self.embeddedSupport, self.searchSupport)
        elif mode == 'settings':
            self.session.openWithCallback(self.subsSetupCB, SubsSetup, self.infobar.subsSettings)
        elif mode == 'settings_embedded':
            self.session.openWithCallback(self.subsSetupEmbeddedCB, SubsSetupEmbedded, self.infobar.subsSettings.embedded)
        elif mode == 'settings_embedded_pli':
            self.session.open(QuickSubtitlesConfigMenu, self.infobar)
        elif mode == 'encoding':
            self.changeEncoding = True
            self.cancel()
        elif mode == 'subsoff':
            self.turnOff = True
            self.cancel()

    def subsChooserCB(self, subfile=None, embeddedSubtitle=None, forceReload=False):
        if subfile is not None and self.subfile != subfile:
            self.subfile = subfile
            self.subdir = os.path.dirname(self.subfile)
            self.newSelection = True
            self.embeddedEnabled = False
            self.cancel()
        elif subfile is not None and self.subfile == subfile and forceReload:
            self.forceReload = True
            self.cancel()
        elif embeddedSubtitle and embeddedSubtitle != self.infobar.selected_subtitle:
            self.embeddedSubtitle = embeddedSubtitle
            self.cancel()
        # self.initGUI()

    def subsSetupCB(self, changedSettings=False, changedEncodingGroup=False,
                    changedShadowType=False, changedEngine=False):
        self.changedSettings = changedSettings
        self.changedEncodingGroup = changedEncodingGroup
        self.changedShadowType = changedShadowType
        self.changedEngine = changedEngine

    def subsSetupEmbeddedCB(self, reloadEmbeddedScreen=False):
        self.reloadEmbeddedScreen = reloadEmbeddedScreen

    def cancel(self):
        self.close(self.subfile, self.embeddedSubtitle, self.changedSettings, self.changeEncoding,
                   self.changedEncodingGroup, self.changedShadowType, self.changedEngine, self.reloadEmbeddedScreen, self.turnOff, self.forceReload)

# rework
class SubsSetup(Screen, ConfigListScreen):
    skin = """
            <screen position="center,center" size="610,435" >
                <widget name="key_red" position="10,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#9f1313" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_green" position="160,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#1f771f" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_yellow" position="310,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#a08500" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_blue" position="460,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#18188b" shadowOffset="-2,-2" shadowColor="black" />
                <eLabel position="-1,55" size="612,1" backgroundColor="#999999" />
                <widget name="config" position="0,75" size="610,360" scrollbarMode="showOnDemand" />
            </screen>"""


    def __init__(self, session, subsSettings):
        Screen.__init__(self, session)
        self.list = [ ]
        ConfigListScreen.__init__(self, self.list, session=session)
        self.subsSettings = subsSettings
        self.setup_title = _("Subtitles setting")

        self["actions"] = ActionMap(["SetupActions", "ColorActions"],
            {
                "cancel": self.keyCancel,
                "green": self.keySave,
                "red": self.keyCancel,
                "blue": self.resetDefaults,
            }, -2)

        self["key_green"] = Label(_("Save"))
        self["key_red"] = Label(_("Cancel"))
        self["key_blue"] = Label(_("Reset Defaults"))
        self["key_yellow"] = Label("")
        self.subsSettings.engine.expert.show.setValue(False)
        self.buildMenu()
        self.onLayoutFinish.append(self.layoutFinished)

    def layoutFinished(self):
        self.setTitle(_("External subtitles setting"))

    def buildMenu(self):
        del self.list[:]
        shadowType = self.subsSettings.external.shadow.type.getValue()
        shadowEnabled = self.subsSettings.external.shadow.enabled.getValue()
        backgroundType = self.subsSettings.external.background.type.getValue()
        backgroundEnabled = self.subsSettings.external.background.enabled.getValue()
        showExpert = self.subsSettings.engine.expert.show.getValue()
        engineType = self.subsSettings.engine.type.getValue()

        self.list.append(getConfigListEntry(_("Pause video on opening subtitles menu"), self.subsSettings.pauseVideoOnSubtitlesMenu))
        self.list.append(getConfigListEntry("-"*120, ConfigNothing()))
        self.list.append(getConfigListEntry(_("Parsing/Rendering"), self.subsSettings.engine.type))
        self.list.append(getConfigListEntry(_("Font type"), self.subsSettings.external.fontType))
        self.list.append(getConfigListEntry(_("Font size"), self.subsSettings.external.fontSize))
        self.list.append(getConfigListEntry(_("Position"), self.subsSettings.external.position))
        self.list.append(getConfigListEntry(_("Color"), self.subsSettings.external.color))
        self.list.append(getConfigListEntry(_("Transparency"), self.subsSettings.external.alpha))
        self.list.append(getConfigListEntry(_("Shadow"), self.subsSettings.external.shadow.enabled))
        if shadowEnabled:
            self.list.append(getConfigListEntry(_("Shadow type"), self.subsSettings.external.shadow.type))
            if shadowType == 'offset':
                self.list.append(getConfigListEntry(_("Shadow X-offset"), self.subsSettings.external.shadow.xOffset))
                self.list.append(getConfigListEntry(_("Shadow Y-offset"), self.subsSettings.external.shadow.yOffset))
            else:
                self.list.append(getConfigListEntry(_("Shadow size"), self.subsSettings.external.shadow.size))
            self.list.append(getConfigListEntry(_("Shadow color"), self.subsSettings.external.shadow.color))
        if engineType == 'standard_new':
            self.list.append(getConfigListEntry(_("Background"), self.subsSettings.external.background.enabled))
            if backgroundEnabled:
                self.list.append(getConfigListEntry(_("Background type"), self.subsSettings.external.background.type))
                if backgroundType == 'dynamic':
                    self.list.append(getConfigListEntry(_("Background X-offset"), self.subsSettings.external.background.xOffset))
                    self.list.append(getConfigListEntry(_("Background Y-offset"), self.subsSettings.external.background.yOffset))
                self.list.append(getConfigListEntry(_("Background color"), self.subsSettings.external.background.color))
                self.list.append(getConfigListEntry(_("Background transparency"), self.subsSettings.external.background.alpha))
        self.list.append(getConfigListEntry(_("Encoding"), self.subsSettings.encodingsGroup))
        self.list.append(getConfigListEntry(_("Show expert settings"), self.subsSettings.engine.expert.show))
        if showExpert:
            self.list.append(getConfigListEntry(_("Hide delay"), self.subsSettings.engine.expert.hideDelay))
            self.list.append(getConfigListEntry(_("Sync delay"), self.subsSettings.engine.expert.syncDelay))
            self.list.append(getConfigListEntry(_("Player delay"), self.subsSettings.engine.expert.playerDelay))
            self.list.append(getConfigListEntry(_("Refresh delay"), self.subsSettings.engine.expert.refreshDelay))
            self.list.append(getConfigListEntry(_("PTS check delay"), self.subsSettings.engine.expert.ptsDelayCheck))
        self["config"].list = self.list
        self["config"].setList(self.list)

    def resetDefaults(self):
        for x in self["config"].list:
            x[1].value = x[1].default
        self.buildMenu()

    def keySave(self):
        changedEngine = self.subsSettings.engine.type.isChanged()
        changedEncodingGroup = self.subsSettings.encodingsGroup.isChanged()
        changedShadowType = self.subsSettings.external.shadow.type.isChanged()
        for x in self["config"].list:
            x[1].save()
        configfile.save()
        self.close(True, changedEncodingGroup, changedShadowType, changedEngine)

    def keyCancel(self):
        for x in self["config"].list:
            x[1].cancel()
        self.close()

    def keyLeft(self):
        ConfigListScreen.keyLeft(self)
        current = self["config"].getCurrent()[1]
        if current in [self.subsSettings.external.shadow.type,
                       self.subsSettings.external.shadow.enabled,
                       self.subsSettings.engine.expert.show,
                       self.subsSettings.engine.type,
                       self.subsSettings.external.background.enabled,
                       self.subsSettings.external.background.type]:
            self.buildMenu()

    def keyRight(self):
        ConfigListScreen.keyRight(self)
        current = self["config"].getCurrent()[1]
        if current in [self.subsSettings.external.shadow.type,
                       self.subsSettings.external.shadow.enabled,
                       self.subsSettings.engine.expert.show,
                       self.subsSettings.engine.type,
                       self.subsSettings.external.background.enabled,
                       self.subsSettings.external.background.type]:
            self.buildMenu()

class SubsSetupEmbedded(Screen, ConfigListScreen):
    skin = """
            <screen position="center,center" size="610,435" >
                <widget name="key_red" position="10,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#9f1313" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_green" position="160,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#1f771f" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_yellow" position="310,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#a08500" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_blue" position="460,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#18188b" shadowOffset="-2,-2" shadowColor="black" />
                <eLabel position="-1,55" size="612,1" backgroundColor="#999999" />
                <widget name="config" position="0,75" size="610,360" scrollbarMode="showOnDemand" />
            </screen>"""

    def __init__(self, session, embeddedSettings):
        Screen.__init__(self, session)
        self.list = [ ]
        self.embeddedSettings = embeddedSettings
        ConfigListScreen.__init__(self, self.list, session=session)

        self["actions"] = ActionMap(["SetupActions", "ColorActions"],
            {
                "cancel": self.keyCancel,
                "green": self.keySave,
                "red": self.keyCancel,
                "blue": self.resetDefaults,
            }, -2)

        self["key_green"] = Label(_("Save"))
        self["key_red"] = Label(_("Cancel"))
        self["key_blue"] = Label(_("Reset Defaults"))
        self["key_yellow"] = Label("")
        self.onLayoutFinish.append(self.layoutFinished)
        self.onLayoutFinish.append(self.buildMenu)

    def layoutFinished(self):
        self.setTitle(_("Embedded subtitles setting"))

    def buildMenu(self):
        fontSizeCfg = getEmbeddedFontSizeCfg(self.embeddedSettings.fontSize)
        self.list.append(getConfigListEntry(_("Font type"), self.embeddedSettings.fontType))
        self.list.append(getConfigListEntry(_("Font size"), fontSizeCfg))
        self.list.append(getConfigListEntry(_("Position"), self.embeddedSettings.position))
        self.list.append(getConfigListEntry(_("Color"), self.embeddedSettings.color))
        self.list.append(getConfigListEntry(_("Shadow X-offset"), self.embeddedSettings.shadow.xOffset))
        self.list.append(getConfigListEntry(_("Shadow Y-offset"), self.embeddedSettings.shadow.yOffset))
        self["config"].list = self.list
        self["config"].setList(self.list)

    def resetDefaults(self):
        for x in self["config"].list:
            x[1].value = x[1].default

    def keySave(self):
        reloadEmbeddedScreen = (self.embeddedSettings.position.isChanged() or
                                   getEmbeddedFontSizeCfg(self.embeddedSettings.fontSize).isChanged())
        for x in self["config"].list:
            x[1].save()
        configfile.save()
        self.close(reloadEmbeddedScreen)

    def keyCancel(self):
        for x in self["config"].list:
            x[1].cancel()
        self.close()

def FileEntryComponent(name, absolute=None, isDir=False):
        res = [ (absolute, isDir) ]
        res.append((eListboxPythonMultiContent.TYPE_TEXT, 35, 1, 470, 20, 0, RT_HALIGN_LEFT, toString(name)))
        if isDir:
            png = LoadPixmap(resolveFilename(SCOPE_SKIN_IMAGE, "extensions/directory.png"))
        else:
            png = LoadPixmap(os.path.join(os.path.dirname(__file__), 'img', 'subtitles.png'))
        if png is not None:
            res.append((eListboxPythonMultiContent.TYPE_PIXMAP_ALPHATEST, 10, 2, 20, 20, png))
        return res



class SubFileList(FileList):
    def __init__(self, defaultDir):
        extensions = []
        for parser in PARSERS:
            extensions += list(parser.parsing)
        FileList.__init__(self, defaultDir, matchingPattern="(?i)^.*\." + '(' + '|'.join(ext[1:] for ext in extensions) + ')', useServiceRef=False)

    def changeDir(self, directory, select=None):
        self.list = []
        # if we are just entering from the list of mount points:
        if self.current_directory is None:
            if directory and self.showMountpoints:
                self.current_mountpoint = self.getMountpointLink(directory)
            else:
                self.current_mountpoint = None
        self.current_directory = directory
        directories = []
        files = []

        if directory is None and self.showMountpoints:  # present available mountpoints
            for p in harddiskmanager.getMountedPartitions():
                path = os_path.join(p.mountpoint, "")
                if path not in self.inhibitMounts and not self.inParentDirs(path, self.inhibitDirs):
                    self.list.append(FileEntryComponent(name=p.description, absolute=path, isDir=True))
            files = [ ]
            directories = [ ]
        elif directory is None:
            files = [ ]
            directories = [ ]
        elif self.useServiceRef:
            root = eServiceReference("2:0:1:0:0:0:0:0:0:0:" + directory)
            if self.additional_extensions:
                root.setName(self.additional_extensions)
            serviceHandler = eServiceCenter.getInstance()
            list = serviceHandler.list(root)

            while 1:
                s = list.getNext()
                if not s.valid():
                    del list
                    break
                if s.flags & s.mustDescent:
                    directories.append(s.getPath())
                else:
                    files.append(s)
            directories.sort()
            files.sort()
        else:
            if os_path.exists(directory):
                files = listdir(directory)
                files.sort()
                tmpfiles = files[:]
                for x in tmpfiles:
                    if os_path.isdir(directory + x):
                        directories.append(directory + x + "/")
                        files.remove(x)

        if directory is not None and self.showDirectories and not self.isTop:
            if directory == self.current_mountpoint and self.showMountpoints:
                self.list.append(FileEntryComponent(name="<" + _("List of Storage Devices") + ">", absolute=None, isDir=True))
            elif (directory != "/") and not (self.inhibitMounts and self.getMountpoint(directory) in self.inhibitMounts):
                self.list.append(FileEntryComponent(name="<" + _("Parent Directory") + ">", absolute='/'.join(directory.split('/')[:-2]) + '/', isDir=True))

        if self.showDirectories:
            for x in directories:
                if not (self.inhibitMounts and self.getMountpoint(x) in self.inhibitMounts) and not self.inParentDirs(x, self.inhibitDirs):
                    name = x.split('/')[-2]
                    self.list.append(FileEntryComponent(name=name, absolute=x, isDir=True))

        if self.showFiles:
            for x in files:
                if self.useServiceRef:
                    path = x.getPath()
                    name = path.split('/')[-1]
                else:
                    path = directory + x
                    name = x

                if (self.matchingPattern is None) or re_compile(self.matchingPattern).search(path):
                    self.list.append(FileEntryComponent(name=name, absolute=x , isDir=False))

        self.l.setList(self.list)

        if select is not None:
            i = 0
            self.moveToIndex(0)
            for x in self.list:
                p = x[0][0]

                if isinstance(p, eServiceReference):
                    p = p.getPath()

                if p == select:
                    self.moveToIndex(i)
                i += 1

class SubsChooserMenuList(MenuList):
    def __init__(self, embeddedAvailable=False, searchSupport=False):
        MenuList.__init__(self, [], False, eListboxPythonMultiContent)
        self.l.setItemHeight(30)
        self.l.setFont(0, gFont("Regular", 20))
        menulist = []
        if embeddedAvailable:
            res = [('embedded')]
            res.append(MultiContentEntryPixmapAlphaTest(pos=(5, 5), size=(35, 25), png=loadPNG(os.path.join(os.path.dirname(__file__), 'img', 'key_red.png'))))
            res.append(MultiContentEntryText(pos=(60, 5), size=(330, 25), font=0, flags=RT_VALIGN_CENTER, text=_("Choose from embedded subtitles")))
            menulist.append(res)
        if searchSupport:
            res = [('search')]
            res.append(MultiContentEntryPixmapAlphaTest(pos=(5, 5), size=(35, 25), png=loadPNG(os.path.join(os.path.dirname(__file__), 'img', 'key_blue.png'))))
            res.append(MultiContentEntryText(pos=(60, 5), size=(330, 25), font=0, flags=RT_VALIGN_CENTER, text=_("Search subtitles")))
            menulist.append(res)
        if embeddedAvailable or searchSupport:
            self.l.setList(menulist)

class E2SubsSeeker(SubsSeeker):
    def __init__(self, session, searchSettings, debug=False):
        self.session = session
        self.download_thread = None
        self.search_settings = searchSettings
        download_path = searchSettings.downloadPath.value
        tmp_path = searchSettings.tmpPath.value

        class SubsSearchSettingsProvider(E2SettingsProvider):
            def __init__(self, providerName, defaults, configSubSection):
                E2SettingsProvider.__init__(self, providerName, configSubSection, defaults)

        SubsSeeker.__init__(self, download_path, tmp_path,
                            captcha_cb=self.captcha_cb,
                            delay_cb=self.delay_cb,
                            message_cb=messageCB,
                            settings_provider_cls=SubsSearchSettingsProvider,
                            settings_provider_args=searchSettings,
                            debug=debug)
        
        self.providers_error = False
        for p in self.seekers:
            if p.error is not None:
                self.providers_error = True

    def downloadSubtitle(self, success_cb, error_cb, selected_subtitle, subtitles_dict, path=None):
        download_fnc = super(E2SubsSeeker, self).downloadSubtitle
        download_settings = {'save_as':self.search_settings.saveAs.value,
                        'lang_to_filename': self.search_settings.addLangToSubsFilename.value,
                        'ask_overwrite': self.search_settings.askOverwriteExistingSubs.value}
        params = (selected_subtitle, subtitles_dict, self.choice_cb, path, self.overwrite_cb, download_settings)
        download_thread = SubsDownloadThread(self.session, download_fnc, params, success_cb, error_cb)
        self.download_thread = download_thread
        self.download_thread.start()

    def _unpack_rarsub(self, rar_path, dest_dir):
        assert self.download_thread is not None
        assert self.download_thread.is_alive()
        files = self.download_thread.getUnrar(rar_path, dest_dir)
        return filter(lambda x:os.path.splitext(x)[1] in ('.srt', '.sub', 'txt'), files)

    def overwrite_cb(self, subfile):
        assert self.download_thread is not None
        assert self.download_thread.is_alive()
        return self.download_thread.getOverwrite(subfile)

    def choice_cb(self, subfiles):
        assert self.download_thread is not None
        assert self.download_thread.is_alive()
        return self.download_thread.getChoice(subfiles)

    def captcha_cb(self, image_path):
        assert self.download_thread is not None
        assert self.download_thread.is_alive()
        return self.download_thread.getCaptcha(image_path)

    def delay_cb(self, seconds):
        assert self.download_thread is not None
        assert self.download_thread.is_alive()
        message = _("Subtitles will be downloaded in") + " " + str(seconds) + " " + _("seconds")
        self.download_thread.getDelay(seconds, message)


class SubsChooser(Screen):
    skin = """
        <screen position="center,center" size="610,460" zPosition="3" >
           <!-- <widget source="filename" render="Label" position="10, 10" size="590,50"  valign="center" halign="center" font="Regular;21" /> -->
            <!-- <eLabel position="5,65" size="600,1" backgroundColor="#999999" /> -->
            <widget name="file_list" position="0,30" size="610,330" scrollbarMode="showOnDemand" />
            <eLabel position="5,370" size="600,1" backgroundColor="#999999" />
            <widget name="menu_list" position="0,380" size="610,80" scrollbarMode="showOnDemand" />
        </screen>
        """

    def __init__(self, session, subsSettings, subdir=None, embeddedSupport=False, searchSupport=False):
        Screen.__init__(self, session)
        self.session = session
        self.subsSettings = subsSettings
        defaultDir = subdir
        if subdir is not None and not subdir.endswith('/'):
            defaultDir = subdir + '/'
        self.embeddedList = None
        self.embeddedSubtitle = None
        if embeddedSupport:
            service = self.session.nav.getCurrentService()
            subtitle = service and service.subtitle()
            self.embeddedList = subtitle and subtitle.getSubtitleList()
        self.searchSupport = searchSupport
        ref = self.session.nav.getCurrentlyPlayingServiceReference()
        videoName = ref and os.path.split(ref.getPath())[1]
        self["filename"] = StaticText(videoName)
        self["file_list"] = SubFileList(defaultDir)
        self["menu_list"] = SubsChooserMenuList(self.embeddedList, searchSupport)
        self["actions"] = NumberActionMap(["OkCancelActions", "ColorActions"],
            {
                "ok": self.ok,
                "cancel": self.close,
                "red": self.embeddedSubsSelection,
                "blue":self.searchSubs
            }, -2)

        self.onLayoutFinish.append(self.updateTitle)
        self.onLayoutFinish.append(self.disableMenuList)

    def updateTitle(self):
        self.setTitle(_("Choose Subtitles"))

    def disableMenuList(self):
        self["menu_list"].selectionEnabled(False)

    def checkEmbeddedSubsSelection(self, embeddedSubtitle=None):
        if embeddedSubtitle:
            self.close(None, embeddedSubtitle)

    def checkDownloadedSubsSelection(self, downloadedSubtitle=None):
        if downloadedSubtitle:
            self.close(downloadedSubtitle, False, True)

    def ok(self):
        if self['file_list'].canDescent():
            self['file_list'].descent()
        else:
            filePath = os.path.join(self['file_list'].current_directory, self['file_list'].getFilename())
            print '[SubsFileChooser]' , filePath
            self.close(filePath, False)

    def embeddedSubsSelection(self):
        if self.embeddedList:
            self.session.openWithCallback(self.checkEmbeddedSubsSelection, SubsEmbeddedSelection)

    def getSearchTitleList(self, sName, sPath):
        searchTitles = []
        if sName:
            searchTitles.append(sName)
        if sPath:
            dirname = os.path.basename(os.path.dirname(sPath))
            dirnameFix = dirname.replace('.', ' ').replace('_', ' ').replace('-', ' ')
            filename = os.path.splitext(os.path.basename(sPath))[0]
            filenameFix = filename.replace('.', ' ').replace('_', ' ').replace('-', ' ')
            if filename not in searchTitles:
                searchTitles.append(filename)
            if filenameFix not in searchTitles:
                searchTitles.append(filenameFix)
            if dirname not in searchTitles:
                searchTitles.append(dirname)
            if dirnameFix not in searchTitles:
                searchTitles.append(dirnameFix)
        return searchTitles

    def searchSubs(self):
        def paramsDialogCB(callback=None):
            if callback:
                self.session.openWithCallback(self.checkDownloadedSubsSelection, SubsSearch, seeker, self.subsSettings.search, sPath, titleList, resetSearchParams=False)
        
        def showProvidersErrorCB(callback):
            if not callback:
                self.subsSettings.search.showProvidersErrorMessage.value = False
            if self.subsSettings.search.openParamsDialogOnSearch.value:
                self.session.openWithCallback(paramsDialogCB, SubsSearchParamsMenu, seeker, self.subsSettings.search, titleList, enabledList=False)
            else:
                self.session.openWithCallback(self.checkDownloadedSubsSelection, SubsSearch, seeker, self.subsSettings.search, sPath, titleList)

        if self.searchSupport:
            ref = self.session.nav.getCurrentlyPlayingServiceReference()
            try:
                sPath = ref.getPath()
            except Exception:
                sPath = None
            try:
                sName = ref.getName()
            except Exception:
                sName = None
            titleList = self.getSearchTitleList(sName, sPath)
            seeker = E2SubsSeeker(self.session, self.subsSettings.search, debug=True)
            
            if seeker.providers_error and self.subsSettings.search.showProvidersErrorMessage.value:
                msg = _("Some subtitles providers are not working") + ".\n"
                msg += _("For more details please check search settings") + "."
                msg += "\n\n"
                msg += _("Do you want to show this message again?")
                self.session.openWithCallback(showProvidersErrorCB, MessageBox, msg, type=MessageBox.TYPE_YESNO)
                
            elif self.subsSettings.search.openParamsDialogOnSearch.value:
                self.session.openWithCallback(paramsDialogCB, SubsSearchParamsMenu, seeker, self.subsSettings.search, titleList, enabledList=False)
            else:
                self.session.openWithCallback(self.checkDownloadedSubsSelection, SubsSearch, seeker, self.subsSettings.search, sPath, titleList)


# source from openpli
class SubsEmbeddedSelection(Screen):
    skin = """<screen name="SubsEmbeddedSelection" position="center,center" size="485,220">
        <widget source="streams" render="Listbox" scrollbarMode="showOnDemand" position="10,40" size="465,180" zPosition="3" transparent="1" >
            <convert type="TemplatedMultiContent">
                {"templates":
                    {"default": (25, [
                        MultiContentEntryText(pos = (0, 0),   size = (35, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 1), # key,
                        MultiContentEntryText(pos = (40, 0),  size = (60, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 2), # number,
                        MultiContentEntryText(pos = (110, 0), size = (120, 25), font = 0, flags = RT_HALIGN_LEFT,  text = 3), # description,
                        MultiContentEntryText(pos = (240, 0), size = (200, 25), font = 0, flags = RT_HALIGN_LEFT,  text = 4), # language,
                    ], True, "showNever"),
                    },
                "fonts": [gFont("Regular", 20), gFont("Regular", 16)],
                "itemHeight": 25
                }
            </convert>
        </widget>
    </screen>"""
    def __init__(self, session):
        Screen.__init__(self, session)
        self["streams"] = List([], enableWrapAround=True)
        self["actions"] = ActionMap(["SetupActions", "DirectionActions", "MenuActions"],
        {
            "ok": self.keyOk,
            "cancel": self.cancel,
        }, -2)
        self.onLayoutFinish.append(self.updateTitle)
        self.onLayoutFinish.append(self.fillList)

    def updateTitle(self):
        self.setTitle(_("Choose subtitles"))

    def fillList(self):
        idx = 0
        streams = []
        subtitlelist = self.getSubtitleList()
        for x in subtitlelist:
            number = str(x[1])
            description = "?"
            language = ""

            try:
                if x[4] != "und":
                    if LanguageCodes.has_key(x[4]):
                        language = LanguageCodes[x[4]][0]
                    else:
                        language = x[4]
            except:
                language = ""

            if x[0] == 0:
                description = "DVB"
                number = "%x" % (x[1])

            elif x[0] == 1:
                description = "teletext"
                number = "%x%02x" % (x[3] and x[3] or 8, x[2])

            elif x[0] == 2:
                types = ("unknown", "embedded", "SSA file", "ASS file",
                        "SRT file", "VOB file", "PGS file")
                try:
                    description = types[x[2]]
                except:
                    description = _("unknown") + ": %s" % x[2]
                number = str(int(number) + 1)
            print x, number, description, language
            streams.append((x, "", number, description, language))
            idx += 1
        self["streams"].list = streams


    def getSubtitleList(self):
        service = self.session.nav.getCurrentService()
        subtitle = service and service.subtitle()
        subtitlelist = subtitle and subtitle.getSubtitleList()
        embeddedlist = []
        for x in subtitlelist:
            if x[0] == 2:
                types = ("unknown", "embedded", "SSA file", "ASS file",
                            "SRT file", "VOB file", "PGS file")
                # filter embedded subtitles
                if x[2] not in [1, 2, 3, 4, 5, 6]:
                    continue
            embeddedlist.append(x)
        return embeddedlist

        self.selectedSubtitle = None
        return subtitlelist

    def cancel(self):
        self.close()

    def keyOk(self):
        cur = self["streams"].getCurrent()
        self.close(cur[0][:4])

class SimpleObserverList(list):
    def __init__(self, *args):
        list.__init__(self, *args)
        self.observer_fncs = list()

    def add_observer(self, observer_fnc):
        self.observer_fncs.append(observer_fnc)

    def remove_observer(self, observer_fnc):
        self.observer_fncs.remove(observer_fnc)

    def append (self, value):
        list.append(self, value)
        for f in self.observer_fncs:
            f()

    def remove(self, value):
        list.remove(self, value)
        for f in self.observer_fncs:
            f()

class SubsDownloadThread(Thread):
    THREADS = SimpleObserverList()

    CAPTCHA_REQUEST = 0
    DELAY_REQUEST = 1
    FINISH_REQUEST_SUCCESS = 2
    FINISH_REQUEST_ERROR = 3
    OVERWRITE_REQUEST = 4
    CHOICE_REQUEST = 5
    UNRAR_REQUEST = 6

    def __init__(self, session, fnc, params, callback, errorback):
        Thread.__init__(self)
        self.session = session
        self.fnc = fnc
        self.params = params
        self.callback = callback
        self.errorback = errorback
        self.messageIn = Queue()
        self.messageOut = Queue()
        self.messagePump = ePythonMessagePump()
        self.messagePump.recv_msg.get().append(self._runInMainThread)

    def start(self):
        SubsDownloadThread.THREADS.append(self)
        Thread.start(self)

    def run(self):
        try:
            ret = self.fnc(*self.params)
            self.messageOut.put((self.FINISH_REQUEST_SUCCESS, ret))
            self.messagePump.send(0)
        except Exception as e:
            self.messageOut.put((self.FINISH_REQUEST_ERROR, e))
            self.messagePump.send(0)

    def _runInMainThread(self, val):
        ret = self.messageOut.get()
        request = ret[0]
        if request == self.CAPTCHA_REQUEST:
            imagePath = ret[1]
            Captcha(self.session, self.getCaptchaCB, imagePath)
        elif request == self.DELAY_REQUEST:
            seconds, message = ret[1], ret[2]
            self.session.openWithCallback(self.getDelayCB, DelayMessageBox, seconds, message)
        elif request == self.FINISH_REQUEST_SUCCESS:
            subFile = ret[1]
            self.callback(subFile)
            SubsDownloadThread.THREADS.remove(self)
        elif request == self.FINISH_REQUEST_ERROR:
            error = ret[1]
            self.errorback(error)
            SubsDownloadThread.THREADS.remove(self)
        elif request == self.CHOICE_REQUEST:
            subFiles = ret[1]
            choiceTitle = _("There are more subtitles in unpacked archive\n please select which one do you want to use")
            choiceList = [(os.path.basename(subfile), subfile) for subfile in subFiles]
            self.session.openWithCallback(self.getChoiceCB, ChoiceBox, choiceTitle, choiceList)
        elif request == self.OVERWRITE_REQUEST:
            overwriteText = _("Subtitles with this name already exist\nDo you want to overwrite them") + "?"
            self.session.openWithCallback(self.getOverwriteCB, MessageBox, overwriteText, MessageBox.TYPE_YESNO)
        elif request == self.UNRAR_REQUEST:
            rarPath = ret[1]
            destDir = ret[2]
            unrar(rarPath, destDir, self.getUnrarCB, self.getUnrarCB)

    def getUnrar(self, subFile, destPath):
        self.messageOut.put((self.UNRAR_REQUEST, subFile, destPath))
        self.messagePump.send(0)
        ret = self.messageIn.get()
        if isinstance(ret, str):
            raise Exception(ret)
        return ret

    def getUnrarCB(self, callback):
        self.messageIn.put(callback)

    def getOverwrite(self, subFile):
        self.messageOut.put((self.OVERWRITE_REQUEST, subFile))
        self.messagePump.send(0)
        return self.messageIn.get()

    def getOverwriteCB(self, callback):
        self.messageIn.put(callback)

    def getChoice(self, files):
        self.messageOut.put((self.CHOICE_REQUEST, files))
        self.messagePump.send(0)
        return self.messageIn.get()

    def getChoiceCB(self, selfile):
        if selfile:
            selfile = selfile[1]
        return self.messageIn.put(selfile)

    def getCaptcha(self, imagePath):
        self.messageOut.put((self.CAPTCHA_REQUEST, imagePath))
        self.messagePump.send(0)
        return self.messageIn.get()

    def getCaptchaCB(self, word):
        word = word or ""
        self.messageIn.put(word)

    def getDelay(self, seconds, message):
        self.messageOut.put((self.DELAY_REQUEST, seconds, message))
        self.messagePump.send(0)
        return self.messageIn.get()

    def getDelayCB(self, callback=None):
        return self.messageIn.put(None)

class SubsSearchProcess(object):
    processes = []
    process_path = os.path.join(os.path.dirname(__file__), 'searchsubs.py')

    def __init__(self):
        self.log = SimpleLogger('SubsSearchProcess', SimpleLogger.LOG_INFO)
        self.toRead = None
        self.pPayload = None
        self.data = ""
        self.__stopping = False
        self.appContainer = eConsoleAppContainer()
        self.appContainer.stdoutAvail.append(self.dataOutCB)
        self.appContainer.stderrAvail.append(self.dataErrCB)
        self.appContainer.appClosed.append(self.finishedCB)

    def recieveMessages(self, data):
        def getMessage(data):
            mSize = int(data[:7])
            mPayload = data[7:mSize]
            mPart = mSize > len(data)
            return mSize, mPayload, mPart

        def readMessage(payload):
            try:
                message = json.loads(payload)
            except EOFError:
                pass
            except Exception:
                self.log.debug('data is not in JSON format! - %s' % str(payload))
            else:
                self.log.debug('message successfully recieved')
                self.toRead = None
                self.pPayload = None
                self.handleMessage(message)

        def readStart(data):
            mSize, mPayload, mPart = getMessage(data)
            if not mPart:
                data = data[mSize:]
                readMessage(mPayload)
                if len(data) > 0:
                    readStart(data)
            else:
                self.toRead = mSize - len(data)
                self.pPayload = mPayload

        def readContinue(data):
            nextdata = data[:self.toRead]
            self.pPayload += nextdata
            data = data[len(nextdata):]
            self.toRead -= len(nextdata)
            if self.toRead == 0:
                readMessage(self.pPayload)
                if len(data) > 0:
                    readStart(data)

        if self.pPayload is not None:
            readContinue(data)
        else:
            readStart(data)

    def handleMessage(self, data):
        self.log.debug('handleMessage "%s"', data)
        if data['message'] == Messages.MESSAGE_UPDATE_CALLBACK:
            self.updateCB(data['value'])
        if data['message'] == Messages.MESSAGE_FINISHED_SCRIPT:
            self.successCB(data['value'])
        if data['message'] == Messages.MESSAGE_CANCELLED_SCRIPT:
            print 'script successfully cancelled'
        if data['message'] == Messages.MESSAGE_ERROR_SCRIPT:
            self.errorCB(data['value'])

    def start(self, params, updateCB, successCB, errorCB):
        self.processes.append(self)
        self.updateCB = updateCB
        self.successCB = successCB
        self.errorCB = errorCB
        cmd = "python %s" % self.process_path
        self.log.debug("start - '%s'", cmd)
        self.appContainer.execute(cmd)
        self.write(params)

    def running(self):
        return self.appContainer.running()

    def stop(self):
        def check_stopped():
            if not self.appContainer.running():
                timer.stop()
                del self.__i
                return
            if self.__i == 0:
                self.__i += 1
                self.log.debug('2. sending SIGKILL')
                self.appContainer.kill()
            elif self.__i == 1:
                timer.stop()
                raise Exception("cannot kill process")

        if self.__stopping:
            self.log.debug('already stopping..')
            return
        self.__stopping = True
        self.log.debug('stopping process..')
        self.__i = 0

        if self.appContainer.running():
            self.log.debug('1. sending SIGINT')
            self.appContainer.sendCtrlC()
            timer = eTimer()
            try:
                      timer_conn = timer.timeout.connect(check_stopped)
            except AttributeError:
                      timer.callback.append(check_stopped)
            
#            timer.callback.append(check_stopped)
            timer.start(2000, False)
        else:
            self.log.debug('process is already stopped')

    def write(self, data):
        dump = json.dumps(data)
        dump = "%07d%s" % (len(dump), dump)
        self.appContainer.write(dump)

    def dataErrCB(self, data):
        self.log.debug("dataErrCB: '%s'", data)
        self.error = data

    def dataOutCB(self, data):
        self.log.debug("dataOutCB: '%s", data)
        self.recieveMessages(data)

    def finishedCB(self, retval):
        self.processes.remove(self)
        self.log.debug('process finished, retval:%d', retval)


class Suggestions(object):
    def __init__(self):
        self._cancelled = False

    def __str__(self):
        return self.__class__.__name__

    def cancel(self):
        self._cancelled = True

    def getSuggestions(self, queryString, successCB, errorCB):
        if queryString is not None:
            d = self._getSuggestions(queryString)
            self.successCB = successCB
            self.errorCB = errorCB
            d.addCallbacks(self.getSuggestionsSuccess, self.getSuggestionsError)

    def getSuggestionsSuccess(self, data):
        print [self], 'success, cancelled', self._cancelled
        if not self._cancelled:
            self.successCB(self._processResult(data))

    def getSuggestionsError(self, failure):
        print [self], 'error, cancelled', self._cancelled
        if not self._cancelled:
            failure.printTraceback()
            self.errorCB(failure)

    def _getSuggestions(self):
        return Deferred()

    def _processResult(self, data):
        return data


class OpenSubtitlesSuggestions(Suggestions):
    def _getSuggestions(self, queryString):
        query = "http://www.opensubtitles.org/libs/suggest.php?format=json2&SubLanguageID=null&MovieName=" + queryString
        return client.getPage(query, timeout=6)

    def _processResult(self, data):
        return json.loads(data)['result']

class HistorySuggestions(Suggestions):
    def __init__(self, historyCfg):
        Suggestions.__init__(self)
        self.historyCfg = historyCfg

    def _getSuggestions(self, queryString):
        def getHistory(queryString):
            historyList = self.historyCfg.value.split(',')
            historyList = [{'name':name, 'total':len(historyList) - idx} for idx, name in enumerate(historyList)]
            d.callback(historyList)
        d = Deferred()
        getHistory(queryString)
        return d

class SuggestionsListScreen(Screen):
    s = getDesktop(0).size()
    desktopSize = (s.width(), s.height())
    windowSize = (int(0.35 * desktopSize[0]), 160)
    skin = """
        <screen name="SuggestionsListScreen" position="%d, 20" zPosition="6" size="%d,160" flags="wfNoBorder" backgroundColor="#33202020" >
            <widget source="suggestionstitle" render="Label" position = "0,0" size="%d, 25" foregroundColor="#00ff00" font="Regular;20" halign="center" valign="center" transparent="0" />
            <widget source="suggestionslist" render="Listbox" position="%d,35" size="%d,115" scrollbarMode="showOnDemand" transparent="1" >
                <convert type="TemplatedMultiContent">
                     {"templates":
                        {"default": (23, [
                            MultiContentEntryText(pos = (0, 0),   size = (360, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 0), # title,
                        ], True, "showOnDemand"),
                        "notselected": (23, [
                            MultiContentEntryText(pos = (0, 0),   size = (360, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 0), # title,
                        ], False, "showOnDemand")
                        },
                    "fonts": [gFont("Regular", 18), gFont("Regular", 16)],
                    "itemHeight": 23
                }
                </convert>
            </widget>
        </screen>""" % (int(0.6 * desktopSize[0]), windowSize[0],
                          windowSize[0],
                          int(0.05 * windowSize[0]), int(0.9 * windowSize[0]))

    def __init__(self, session, title, configTextWithSuggestions):
        Screen.__init__(self, session)
        self.activeState = False
        self.list = []
        self.suggestlist = []
        self["suggestionslist"] = List(self.list)
        self["suggestionstitle"] = StaticText(title.encode('utf-8'))
        self.configTextWithSuggestion = configTextWithSuggestions

    def update(self, suggestions):
        if suggestions and len(suggestions) > 0:
            if not self.shown:
                self.show()
            suggestions.sort(key=lambda x: int(x['total']))
            suggestions.reverse()
            if len(suggestions):
                self.list = []
                for s in suggestions:
                    self.list.append((s['name'].encode('utf-8'),))
                self["suggestionslist"].setList(self.list)
                self["suggestionslist"].setIndex(0)
        else:
            self.hide()

    def getlistlenght(self):
        return len(self.list)

    def up(self):
        if self.list and len(self.list) > 0:
            self["suggestionslist"].selectPrevious()
            return self.getSelection()

    def down(self):
        if self.list and len(self.list) > 0:
            self["suggestionslist"].selectNext()
            return self.getSelection()

    def pageUp(self):
        if self.list and len(self.list) > 0:
            self["suggestionslist"].selectPrevious()
            return self.getSelection()

    def pageDown(self):
        if self.list and len(self.list) > 0:
            self["suggestionslist"].selectNext()
            return self.getSelection()

    def activate(self):
        self.activeState = True
        self.enableSelection(True)
        return self.getSelection()

    def deactivate(self):
        self.activeState = False
        self.enableSelection(False)
        return self.getSelection()

    def getSelection(self):
        if self["suggestionslist"].getCurrent() is None:
            return None
        return self["suggestionslist"].getCurrent()[0]

    def enableSelection(self, value):
        if value:
            self['suggestionslist'].style = 'default'
        else:
            self['suggestionslist'].style = 'notselected'

class HistoryListScreen(SuggestionsListScreen):
    s = getDesktop(0).size()
    desktopSize = (s.width(), s.height())
    windowSize = (int(0.35 * desktopSize[0]), 160)
    skin = """
        <screen name="HistoryListScreen" position="%d, 20" zPosition="6" size="%d,160" flags="wfNoBorder" backgroundColor="#33202020" >
            <widget source="suggestionstitle" render="Label" position = "0,0" size="%d, 25" foregroundColor="#ff0000" font="Regular;20" halign="center" valign="center" transparent="0" />
            <widget source="suggestionslist" render="Listbox" position="%d,35" size="%d,115" scrollbarMode="showOnDemand" transparent="1" >
                <convert type="TemplatedMultiContent">
                     {"templates":
                        {"default": (23, [
                            MultiContentEntryText(pos = (0, 0),   size = (360, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 0), # title,
                        ], True, "showOnDemand"),
                        "notselected": (23, [
                            MultiContentEntryText(pos = (0, 0),   size = (360, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 0), # title,
                        ], False, "showOnDemand")
                        },
                    "fonts": [gFont("Regular", 18), gFont("Regular", 16)],
                    "itemHeight": 23
                }
                </convert>
            </widget>
        </screen>""" % (int(0.05 * desktopSize[0]), windowSize[0],
                          windowSize[0],
                          int(0.05 * windowSize[0]), int(0.9 * windowSize[0]))

    def __init__(self, session, title, configTextWithSuggestions):
        SuggestionsListScreen.__init__(self, session, title, configTextWithSuggestions)

class ConfigTextWithSuggestionsAndHistory(ConfigText):
    def __init__(self, historyCfg, default="", fixed_size=True, visible_width=False):
        ConfigText.__init__(self, default, fixed_size, visible_width)
        self.historyCfg = historyCfg
        self.historyClass = HistorySuggestions
        self.__history = None
        self.suggestionsClass = OpenSubtitlesSuggestions
        self.__suggestions = None
        self.currentWindow = None

    def handleKey(self, key):
        ConfigText.handleKey(self, key)
        if key in [KEY_DELETE, KEY_BACKSPACE, KEY_ASCII, KEY_TIMEOUT]:
            self.getSuggestions()

    def onSelect(self, session):
        ConfigText.onSelect(self, session)
        if session is not None:
            suggestionsWindowTitle = _("Suggestions") + " (" + _("press green") + ")"
            self.suggestionsWindow = session.instantiateDialog(SuggestionsListScreen, suggestionsWindowTitle , self)
            self.suggestionsWindow.deactivate()
            self.suggestionsWindow.show()
            historyWindowTitle = _("History") + " (" + _("press red") + ")"
            self.historyWindow = session.instantiateDialog(HistoryListScreen, historyWindowTitle, self)
            self.historyWindow.deactivate()
            self.historyWindow.show()
        self.getSuggestions()
        self.getHistory()

    def onDeselect(self, session):
        self.cancelGetSuggestions()
        self.cancelGetHistory()
        ConfigText.onDeselect(self, session)
        if self.suggestionsWindow:
            session.deleteDialog(self.suggestionsWindow)
            self.suggestionsWindow = None
        if self.historyWindow:
            session.deleteDialog(self.historyWindow)
            self.historyWindow = None

    def getCurrentSelection(self):
        if self.currentWindow.getlistlenght() > 0:
            return self.currentWindow.getSelection()

    def currentListUp(self):
        if self.currentWindow.getlistlenght() > 0:
            self.value = self.currentWindow.up()

    def currentListDown(self):
        if self.currentWindow.getlistlenght() > 0:
            self.value = self.currentWindow.down()

    def currentListPageDown(self):
        if self.currentWindow.getlistlenght() > 0:
            self.value = self.currentWindow.pageDown()

    def currentListPageUp(self):
        if self.currentWindow.getlistlenght() > 0:
            self.value = self.currentWindow.pageUp()

    def propagateSuggestions(self, suggestionsList):
        self.cancelGetSuggestions()
        if self.suggestionsWindow:
            self.suggestionsWindow.update(suggestionsList)

    def propagateHistory(self, historyList):
        self.cancelGetHistory()
        if self.historyWindow:
            self.historyWindow.update(historyList)

    def enableSuggestions(self, value):
        if value:
            if self.suggestionsWindow:
                self.tmpValue = self.value
                selection = self.suggestionsWindow.activate()
                if selection is None:
                    print 'empty suggesstions list'
                    return False
                self.value = selection
                self.currentWindow = self.suggestionsWindow
                return True
            else:
                print 'Error - suggestionsWindow no longer exists'
                return False
        else:
            self.cancelGetSuggestions()
            if self.suggestionsWindow:
                self.suggestionsWindow.deactivate()
                self.currentWindow = None
                self.getSuggestions()
                return True
            else:
                print 'Error - suggestionsWindow no longer exists'
                return False

    def enableHistory(self, value):
        if value:
            if self.historyWindow:
                self.tmpValue = self.value
                selection = self.historyWindow.activate()
                if selection is None:
                    print "Error - empty history list"
                    return False
                self.value = selection
                self.currentWindow = self.historyWindow
                return True
            else:
                print 'Error - historyWindow no longer exists'
                return False
        else:
            self.cancelGetHistory()
            if self.historyWindow:
                self.historyWindow.deactivate()
                self.currentWindow = None
                self.getHistory()
                return True
            else:
                print 'Error - historyWindow no longer exists'
                return False

    def cancelGetSuggestions(self):
        if self.__suggestions is not None:
            self.__suggestions.cancel()

    def cancelGetHistory(self):
        if self.__history is not None:
            self.__history.cancel()

    def gotSuggestionsError(self, val):
        print "[ConfigTextWithSuggestions] gotSuggestionsError:", val

    def gotHistoryError(self, val):
        print "[ConfigTextWithSuggestions] gotHistoryError:", val

    def getSuggestions(self):
        self.__suggestions = self.suggestionsClass().getSuggestions(self.value, self.propagateSuggestions, self.gotSuggestionsError)

    def getHistory(self):
        self.__history = self.historyClass(self.historyCfg).getSuggestions(self.value, self.propagateHistory, self.gotHistoryError)

    def cancelSuggestions(self):
        self.value = self.tmpValue
        self.enableSuggestions(False)
        self.enableHistory(False)

class Message(object):
    def __init__(self, infowidget, errorwidget):
        self.infowidget = infowidget
        self.errorwidget = errorwidget
        self.timer = eTimer()
        try:
                      self.timer_conn = self.timer.timeout.connect(self.hide)
        except AttributeError:
                      self.timer.callback.append(self.hide)
        
#        self.timer.callback.append(self.hide)

    def info(self, text, timeout=None):
        self.timer.stop()
        self.errorwidget.hide()
        self.infowidget.setText(text)
        self.infowidget.show()
        if timeout:
            self.timer.start(timeout, True)

    def error(self, text, timeout=None):
        self.timer.stop()
        self.infowidget.hide()
        self.errorwidget.setText(text)
        self.errorwidget.show()
        if timeout:
            self.timer.start(timeout, True)

    def hide(self):
        self.timer.stop()
        self.errorwidget.hide()
        self.infowidget.hide()

class SearchParamsHelper(object):
    def __init__(self, seeker, searchSettings):
        self.seeker = seeker
        self.searchSettings = searchSettings
        self.searchTitle = searchSettings.title
        self.searchType = searchSettings.type
        self.searchYear = searchSettings.year
        self.searchSeason = searchSettings.season
        self.searchEpisode = searchSettings.episode
        self.searchProvider = searchSettings.provider
        self.searchUseFilePath = searchSettings.useFilePath

    def resetSearchParams(self):
        self.searchType.value = self.searchType.default
        self.searchType.save()
        self.searchTitle.value = self.searchTitle.default
        self.searchTitle.save()
        self.searchSeason.value = self.searchSeason.default
        self.searchSeason.save()
        self.searchEpisode.value = self.searchEpisode.default
        self.searchEpisode.save()
        self.searchYear.value = self.searchYear.default
        self.searchYear.save()
        self.searchProvider.value = self.searchProvider.default
        self.searchProvider.save()
        self.searchUseFilePath.value = self.searchUseFilePath.default

    def detectSearchParams(self, filepath="", searchExpression=""):
        self.resetSearchParams()
        params = detectSearchParams(filepath, searchExpression)
        if params[2]:
            self.searchType.value = "tv_show"
            self.searchTitle.value = params[2]
            self.searchSeason.value = params[3] and int(params[3]) or 0
            self.searchEpisode.value = params[4] and int(params[4]) or 0
        else:
            self.searchType.value = "movie"
            self.searchTitle.value = params[0]
            self.searchYear.value = params[1] and int(params[1]) or 0
        self.updateProviders()

    def getSearchParams(self):
        langs = [self.searchSettings.lang1.value,
                 self.searchSettings.lang2.value,
                 self.searchSettings.lang3.value]
        provider = self.searchProvider.value
        title = self.searchTitle.value
        tvshow = self.searchType.value == "tv_show" and title or ""
        year = self.searchYear.value and not tvshow and str(self.searchYear.value) or ""
        season = self.searchSeason.value and tvshow and str(self.searchSeason.value) or ""
        episode = self.searchEpisode.value and tvshow and str(self.searchEpisode.value) or ""
        return provider, langs, title, year, tvshow, season, episode

    def updateProviders(self):
        tvshow = self.searchType.value == "tv_show"
        providers = self.seeker.getProviders([self.searchSettings.lang1.value,
                 self.searchSettings.lang2.value,
                 self.searchSettings.lang3.value], not tvshow, tvshow)
        choiceList = []
        choiceList.append(("all", _("All")))
        choiceList.extend((p.id, p.provider_name) for p in providers)
        self.searchProvider.setChoices(choiceList)
        if self.searchProvider.value not in [p.id for p in providers]:
            if tvshow:
                self.searchProvider.value = self.searchSettings.tvshowProvider.value
            else:
                self.searchProvider.value = self.searchSettings.movieProvider.value
        tvshowProviders = self.seeker.getProviders(movie=False, tvshow=True)
        choiceList = []
        choiceList.append(("all", _("All")))
        choiceList.extend((p.id, p.provider_name) for p in tvshowProviders)
        self.searchSettings.tvshowProvider.setChoices(choiceList)
        movieProviders = self.seeker.getProviders(movie=True, tvshow=False)
        choiceList = []
        choiceList.append(("all", _("All")))
        choiceList.extend((p.id, p.provider_name) for p in movieProviders)
        self.searchSettings.movieProvider.setChoices(choiceList)

class SubsSearch(Screen):
    skin = """
    <screen name="SubsSearch" position="center,center" size="700,520" zPosition="3" >
        <widget source="languages" render="Listbox" position="10,10" size="680,150" zPosition="3" scrollbarMode="showNever"  transparent="1" >
            <convert type="TemplatedMultiContent">
                {"templates":
                    {"default": (20, [
                        MultiContentEntryText(pos = (0, 0),   size = (200, 20),  font = 0, color = 0xDAA520, flags = RT_HALIGN_LEFT,  text = 0), # langname,
                        MultiContentEntryText(pos = (205, 0),   size = (400, 20),  font = 0, flags = RT_HALIGN_LEFT,  text = 1)
                    ], False, "showNever"),
                    },
                "fonts": [gFont("Regular", 18), gFont("Regular", 16)],
                "itemHeight":20,
                }
            </convert>
        </widget>
        <widget source="subtitles_header" render="Listbox" scrollbarMode="showNever" position="5,175" size="690,25" zPosition="3" transparent="1" >
            <convert type="TemplatedMultiContent">
                {"templates":
                    {"default": (23, [
                        MultiContentEntryText(pos = (0, 0),   size = (120, 25),  font = 0, flags = RT_HALIGN_LEFT,  color=0xcccccc, text = 0), # language,
                        MultiContentEntryText(pos = (140, 0),  size = (335, 25),  font = 0, flags = RT_HALIGN_LEFT,  color=0xcccccc, text = 1), # filename,
                        MultiContentEntryText(pos = (500, 0), size = (135, 25), font = 0, flags = RT_HALIGN_LEFT,  color=0xcccccc, text = 2), # provider,
                        MultiContentEntryText(pos = (645, 0), size = (20, 25), font = 0, flags = RT_HALIGN_LEFT,  color=0xcccccc, text = 4), # sync,
                    ], False, "showOnDemand"),
                    "old": (23, [
                        MultiContentEntryText(pos = (0, 0),   size = (90, 25),  font = 0, flags = RT_HALIGN_LEFT,  color=0xcccccc, text = 0), # language,
                        MultiContentEntryText(pos = (100, 0),  size = (335, 25),  font = 0, flags = RT_HALIGN_LEFT,  color=0xcccccc, text = 1), # filename,
                        MultiContentEntryText(pos = (445, 0), size = (130, 25), font = 0, flags = RT_HALIGN_LEFT,  color=0xcccccc, text = 2), # provider,
                        MultiContentEntryText(pos = (585, 0), size = (95, 25), font = 0, flags = RT_HALIGN_LEFT,  color=0xcccccc, text = 3), # size,
                    ], False, "showOnDemand"),
                    },
                "fonts": [gFont("Regular", 18), gFont("Regular", 16)],
                "itemHeight": 23
                }
            </convert>
        </widget>
        <eLabel position="5,205" size="690,1" backgroundColor="#999999" />
        <widget name="loadmessage"  position="5,210" size="690,260" valign="center" halign="center" font="Regular;19" foregroundColor="#ffffff" zPosition="4" />
        <widget name="errormessage" position="5,210" size="690,260" valign="center" halign="center" font="Regular;19" foregroundColor="#ff0000" zPosition="5" />
        <widget source="subtitles" render="Listbox" scrollbarMode="showOnDemand" position="5,210" size="690,260" zPosition="3" transparent="1" >
            <convert type="TemplatedMultiContent">
                {"templates":
                    {"default": (23, [
                        MultiContentEntryPixmapAlphaBlend(pos = (0, 0),   size = (24, 24), png=0), # key,
                        MultiContentEntryText(pos = (30, 0),   size = (100, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 1), # language,
                        MultiContentEntryText(pos = (140, 0),  size = (335, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 2), # filename,
                        MultiContentEntryText(pos = (500, 0), size = (135, 25), font = 0, flags = RT_HALIGN_LEFT,  text = 3), # size,
                        MultiContentEntryPixmapAlphaBlend(pos = (645, 0),   size = (24, 24), png=4), # syncPng,
                    ], True, "showOnDemand"),
                    "old": (23, [
                        MultiContentEntryPixmapAlphaBlend(pos = (0, 0),   size = (24, 24), png=0), # key,
                        MultiContentEntryText(pos = (30, 0),   size = (60, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 1), # language,
                        MultiContentEntryText(pos = (100, 0),  size = (335, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 2), # filename,
                        MultiContentEntryText(pos = (445, 0), size = (130, 25), font = 0, flags = RT_HALIGN_LEFT,  text = 3), # size,
                        MultiContentEntryText(pos = (585, 0), size = (95, 25), font = 0, flags = RT_HALIGN_LEFT,  text = 4), # sync,
                    ], True, "showOnDemand"),
                    },
                "fonts": [gFont("Regular", 18), gFont("Regular", 16)],
                "itemHeight": 23
                }
            </convert>
        </widget>
        <eLabel position="5,475" size="690,1" backgroundColor="#999999" />
        <widget source="bottom_menu" render="Listbox" position="10,485" size="680,180" zPosition="3" scrollbarMode="showNever" transparent="1" >
            <convert type="TemplatedMultiContent">
                {"template": [
                        MultiContentEntryPixmapAlphaBlend(pos = (0, 0),   size = (35, 25), png=0), # key,
                        MultiContentEntryText(pos = (40, 0),  size = (170, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 1), # number,
                        MultiContentEntryPixmapAlphaBlend(pos = (245, 0),   size = (35, 25), png=2), # key,
                        MultiContentEntryText(pos = (285, 0),  size = (170, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 3), # number,
                        MultiContentEntryPixmapAlphaBlend(pos = (470, 0),   size = (35, 25), png=4), # key,
                        MultiContentEntryText(pos = (510, 0), size = (170, 25), font = 0, flags = RT_HALIGN_LEFT,  text = 5), # description,
                    ],
                    "fonts": [gFont("Regular", 20), gFont("Regular", 16)],
                    "itemHeight": 25,
                    "selectionEnabled":False
                }
            </convert>
        </widget>
    </screen> """

    def __init__(self, session, seeker, searchSettings, filepath=None, searchTitles=None, streamed=None, resetSearchParams=True):
        Screen.__init__(self, session)
        self.searchSettings = searchSettings
        filepath = filepath or ""
        self.streamed = streamed is not None and streamed or filepath.startswith(('http', 'rtmp', 'mms', 'hds', 'hls'))
        searchTitles = searchTitles or [""]
        self.searchParamsHelper = SearchParamsHelper(seeker, searchSettings)
        self.seeker = seeker
        self.searchExpression = searchTitles[0]
        self.searchTitles = searchTitles
        self.filepath = filepath
        self.searchTitle = searchSettings.title
        self.searchType = searchSettings.type
        self.searchYear = searchSettings.year
        self.searchSeason = searchSettings.season
        self.searchEpisode = searchSettings.episode
        self.searchProvider = searchSettings.provider
        self.searchUseFilePath = searchSettings.useFilePath
        self.lastSearchParams = None
        self['loadmessage'] = Label("")
        self['errormessage'] = Label("")
        self['movieinfo'] = List([])
        self['searchstatus'] = List([])
        self['languages'] = List([])
        self['subtitles_header'] = List([])
        self['subtitles'] = List([])
        self['bottom_menu'] = List([])
        self["actions"] = ActionMap(["ColorActions", "DirectionActions", "OkCancelActions"],
        {
            "ok": self.keyOk,
            "cancel": self.keyCancel,
            "up": self.keyUp,
            "upRepeated": self.keyUp,
            "down": self.keyDown,
            "downRepeated": self.keyDown,
            "right":self.keyRight,
            "rightRepeated":self.keyRight,
            "left":self.keyLeft,
            "leftRepeated":self.keyLeft,
            "red":self.keyRed,
            "green":self.keyGreen,
            "yellow":self.keyYellow,
            "blue":self.keyBlue,
        }, -2)
        self.message = Message(self['loadmessage'], self['errormessage'])
        self.working = False
        self.onLayoutFinish.append(self.updateTitle)
        if resetSearchParams:
            self.onLayoutFinish.append(self.detectSearchParams)
            self.onLayoutFinish.append(self.updateProviders)
        self.onLayoutFinish.append(self.updateSearchInfoList)
        self.onLayoutFinish.append(self.updateSubtitlesHeaderList)
        self.onLayoutFinish.append(self.updateBottomList)
        if not searchSettings.manualSearch.value:
            self.onLayoutFinish.append(self.searchSubs)
        else:
            self.onLayoutFinish.append(self.searchMessage)
        self.onClose.append(self.message.hide)
        self.onClose.append(self.resetSearchParams)
        self.onClose.append(self.cancelSearch)
        self.onClose.append(self.closeSeekers)

    def updateTitle(self):
        self.title = _("Subtitles search")

    def updateProviders(self):
        self.searchParamsHelper.updateProviders()

    def updateSearchInfoList(self):
        searchInfoList = []
        lang1 = self.searchSettings.lang1.value
        lang2 = self.searchSettings.lang2.value
        lang3 = self.searchSettings.lang3.value
        lang1 = lang1 in LanguageCodes and LanguageCodes[lang1][0] or lang1
        lang2 = lang2 in LanguageCodes and LanguageCodes[lang2][0] or lang2
        lang3 = lang3 in LanguageCodes and LanguageCodes[lang3][0] or lang3
        languages = ", ".join(_(lang) for lang in set([lang1, lang2, lang3]))
        year = self.searchYear.value and str(self.searchYear.value) or ""
        season = self.searchSeason.value and str(self.searchSeason.value) or ""
        episode = self.searchEpisode.value and str(self.searchEpisode.value) or ""
        useFilePathStr = self.searchUseFilePath.value and _("yes") or _("no")
        searchInfoList.append((_("Title") + ":", self.searchTitle.value))
        searchInfoList.append((_("Type") + ":", self.searchType.getText()))
        if self.searchType.value == "movie":
            searchInfoList.append((_("Year") + ":", year))
        else:
            searchInfoList.append((_("Season") + ":", season))
            searchInfoList.append((_("Episode") + ":", episode))
        searchInfoList.append((_("Provider") + ":", self.searchProvider.getText()))
        searchInfoList.append((_("Preferred languages") + ":", languages))
        searchInfoList.append((_("Use File path") + ":", useFilePathStr))
        self['languages'].list = searchInfoList

    def updateSubtitlesHeaderList(self):
        header = [(_("Language"), _("Release"), _("Provider"), _("Size"), _("S")), ]
        self['subtitles_header'].list = header

    def updateSubsList(self):
        imgDict = {'sync':loadPNG(os.path.join(os.path.dirname(__file__), 'img', 'check.png')),
                             'unk':loadPNG(os.path.join(os.path.dirname(__file__), 'img', 'countries',  'UNK.png'))}
        subtitleListGUI = []
        for sub in self.subtitlesList:
            sync = 'sync' in sub and sub['sync'] or False
            if sub['country'] not in imgDict:
                countryImgPath = os.path.join(os.path.dirname(__file__), 'img', 'countries', sub['country'] + '.png')
                if os.path.isfile(countryImgPath):
                    imgDict[sub['country']] = loadPNG(os.path.join(os.path.dirname(__file__), 'img', 'countries', sub['country'] + '.png'))
                    countryPng = imgDict[sub['country']]
                else:
                    countryPng = imgDict['unk']
            syncPng = sync and imgDict['sync'] or None
            subtitleListGUI.append((countryPng, _(toString(sub['language_name'])), toString(sub['filename']), toString(sub['provider']), syncPng),)
        imgDict = None
        self['subtitles'].list = subtitleListGUI

    def updateBottomList(self):
        redPng = loadPNG(os.path.join(os.path.dirname(__file__), 'img', 'key_red.png'))
        redText = _("Update Expression")
        greenPng = loadPNG(os.path.join(os.path.dirname(__file__), 'img', 'key_green.png'))
        greenText = _("Search")
        yellowPng = loadPNG(os.path.join(os.path.dirname(__file__), 'img', 'key_yellow.png'))
        yellowText = _("Settings")
        self['bottom_menu'].list = [(redPng, redText, greenPng, greenText, yellowPng, yellowText), ]

    def resetSearchParams(self):
        self.searchParamsHelper.resetSearchParams()

    def detectSearchParams(self):
        self.searchParamsHelper.detectSearchParams(self.filepath, self.searchExpression)

    def getSearchParams(self):
        return self.searchParamsHelper.getSearchParams()

    def closeSeekers(self):
        for seeker in self.seeker.seekers:
            seeker.close()

    def keyOk(self):
        if not self.working and self['subtitles'].count():
            self.working = True
            self.downloadSubs(self.subtitlesList[self["subtitles"].index])

    def keyCancel(self):
        if not self.working:
            self.close()

    def keyUp(self):
        if not self.working and self['subtitles'].count():
            self.message.hide()
            self['subtitles'].selectPrevious()

    def keyDown(self):
        if not self.working and self['subtitles'].count():
            self.message.hide()
            self['subtitles'].selectNext()

    def __getRenderer(self, source):
        while not isinstance(source, Renderer):
            source = source.downstream_elements[0]
        return source

    def keyRight(self):
        if not self.working and self['subtitles'].count():
            self.message.hide()
            # hack
            listbox = self.__getRenderer(self['subtitles'])
            listbox.move(listbox.instance.pageDown)

    def keyLeft(self):
        if not self.working and self['subtitles'].count():
            self.message.hide()
            # hack
            listbox = self.__getRenderer(self['subtitles'])
            listbox.move(listbox.instance.pageUp)

    def keyRed(self):
        if not self.working:
            self.updateSearchParams()
            print 'change search expression'

    def keyGreen(self):
        if not self.working:
            self.search()
            print 'search'

    def keyYellow(self):
        if not self.working:
            self.openSettings(self.seeker)
            print 'subtitles settings'

    def keyBlue(self):
        if not self.working:
            pass

    def cancelSearch(self):
        for p in SubsSearchProcess.processes:
            p.stop()
        print len(SubsSearchProcess.processes), 'processes still running'

    def searchMessage(self):
        self.message.info(_("Update search parameters if not correct\n and press green button for search"))

    def searchSubs(self):
        finished = []
        def searchSubsUpdate(args):
            pFinished = args[0]
            finished.append(pFinished)
            progressMessage = "%s - %d%%" % (_("loading subtitles list"), int(len(finished) / float(len(provider)) * 100))
            self.message.info(progressMessage)

        self.cancelSearch()
        self.subtitlesList = []
        self.subtitlesDict = {}
        p = self.lastSearchParams = self.getSearchParams()
        langs, title, year, tvshow, season, episode = p[1], p[2], p[3], p[4], p[5], p[6]
        providers = self.seeker.getProviders(langs, not tvshow, tvshow)
        if self.searchProvider.value == "all":
            provider = providers
        else:
            provider = [p for p in providers
                        if p.id == self.searchProvider.value]
        filePath = self.searchUseFilePath.value and self.filepath or None
        timeout = float(self.searchSettings.timeout.value)
        params = {
                  'search':{
                            'providers':[p.id for p in provider],
                            'title':title,
                            'filepath':filePath,
                            'langs':langs,
                            'year': year,
                            'tvshow': tvshow,
                            'season': season,
                            'episode': episode,
                            'timeout': timeout
                            },
                  'settings': dict((s.id, s.settings_provider.getSettingsDict()) for s in self.seeker.seekers)
                  }
        searchProcess = SubsSearchProcess()
        self.message.info("%s - %d%%" % (_("loading subtitles list"), 0))
        self.updateSubsList()
        self.updateBottomList()
        searchProcess.start(params, searchSubsUpdate, self.searchSubsSuccess, self.searchSubsError)

    def searchSubsSuccess(self, subtitles):
        print '[SubsSearch] search success'
        self.message.hide()
        self.subtitlesDict = subtitles
        subtitlesList = self.seeker.getSubtitlesList(subtitles)
        subtitlesList = self.seeker.sortSubtitlesList(subtitlesList, sort_sync=True)
        langs = [self.searchSettings.lang1.value,
                    self.searchSettings.lang2.value,
                    self.searchSettings.lang3.value]
        if self.searchSettings.defaultSort.value == 'lang':
            subtitlesList = self.seeker.sortSubtitlesList(subtitlesList, langs, sort_langs=True)
        elif self.searchSettings.defaultSort.value == 'provider':
            subtitlesList = self.seeker.sortSubtitlesList(subtitlesList, langs, sort_provider=True)
        self.subtitlesList = subtitlesList
        if len(self.subtitlesList) == 0:
            noSubtitlesMessage = _("No subtitles found :(")
            noSubtitlesMessage += "\n" + _("Try update(simplify) search expression and try again..")
            self.message.info(noSubtitlesMessage)
        self.updateSubsList()
        self.working = False

    def searchSubsError(self, error):
        print '[SubsSearch] search error', str(error)
        self.message.error(error.message, 4000)
        self.subtitlesList = []
        self.subtitlesDict = {}
        self.updateSubsList()

    def downloadSubs(self, subtitle):
        def download(path):
            self.working = True
            self.message.info(_('downloading subtitles...'))
            self.seeker.downloadSubtitle(self.downloadSubsSuccess, self.downloadSubsError, subtitle, self.subtitlesDict, path)

        def askForDownloadDirCB(downloadDir):
            if downloadDir:
                download(downloadDir)
            else: self.working = False

        if self.searchSettings.saveTo.value == 'video' and not self.streamed:
            downloadDir = os.path.dirname(self.filepath)
        else:
            downloadDir = self.searchSettings.downloadPath.value
        if self.searchSettings.askForDownloadLocation.value:
            locationBoxText = _("Select download location")
            self.session.openWithCallback(askForDownloadDirCB, LocationBox, locationBoxText, currDir=downloadDir)
        else:
            download(downloadDir)

    def downloadSubsSuccess(self, subFile):
        print '[SubsSearch] download success %s' % str(subFile)
        self.working = False
        self.close(subFile)

    def downloadSubsError(self, e):
        print '[SubsSearch] download error', str(e)
        self.working = False
        errorMessageFormat = "[{0}]: {1}"
        if isinstance(e, SubtitlesDownloadError):
            if e.code == SubtitlesErrors.CAPTCHA_RETYPE_ERROR:
                self.message.error(errorMessageFormat.format(e.provider, _("captcha doesn't match, try again...")), 4000)
            elif e.code == SubtitlesErrors.INVALID_CREDENTIALS_ERROR:
                self.message.error(errorMessageFormat.format(e.provider, _("invalid credentials provided, correct them and try again")), 4000)
            elif e.code == SubtitlesErrors.NO_CREDENTIALS_ERROR:
                self.message.error(errorMessageFormat.format(e.provider, _("no credentials provided, set them and try again")), 4000)
            else:
                self.message.error(str(e), 4000)
        else:
            self.message.error(str(e), 4000)

    def toggleSearchExpression(self):
        currIdx = self.searchTitles.index(self.searchExpression)
        if self.searchExpression == self.searchTitles[-1]:
            currIdx = 0
        else: currIdx += 1
        self.searchExpression = self.searchTitles[currIdx]

    def updateSearchParams(self):
        self.session.openWithCallback(self.updateSearchParamsCB, SubsSearchParamsMenu, self.seeker, self.searchSettings, self.searchTitles, False)

    def updateSearchParamsCB(self, callback=None):
        if callback:
            self.updateSearchInfoList()
            self.updateBottomList()
            if not self.searchSettings.manualSearch.value:
                self.search()

    def search(self):
        self.searchSubs()

    def openSettings(self, seekers):
        self.session.openWithCallback(self.openSettingsCB, SubsSearchSettings, self.searchSettings, seekers, self.streamed)

    def openSettingsCB(self, langChanged=False):
            self.seeker.tmp_path = self.searchSettings.tmpPath.value
            self.seeker.download_path = self.searchSettings.downloadPath.value
            self.updateProviders()
            self.updateSearchInfoList()
            self.updateBottomList()
            if langChanged and not self.searchSettings.manualSearch.value:
                self.searchSubs()


class SubsSearchSettings(Screen, ConfigListScreen):
    skin = """<screen name="SubsSearch" position="center,center" size="650,500" zPosition="3" >
        <widget name="key_red" position="10,5" zPosition="1" size="150,45" font="Regular;20" halign="center" valign="center" backgroundColor="#9f1313" shadowOffset="-2,-2" shadowColor="black" />
        <widget name="key_green" position="170,5" zPosition="1" size="150,45" font="Regular;20" halign="center" valign="center" backgroundColor="#1f771f" shadowOffset="-2,-2" shadowColor="black" />
        <widget name="key_yellow" position="330,5" zPosition="1" size="150,45" font="Regular;20" halign="center" valign="center" backgroundColor="#a08500" shadowOffset="-2,-2" shadowColor="black" />
        <widget name="key_blue" position="490,5" zPosition="1" size="150,45" font="Regular;20" halign="center" valign="center" backgroundColor="#18188b" shadowOffset="-2,-2" shadowColor="black" />
        <eLabel position="-1,55" size="650,1" backgroundColor="#999999" />
        <widget name="config" position="10,75" size="630,160" scrollbarMode="showOnDemand" />
        <widget source="providers_header" render="Listbox" scrollbarMode="showOnDemand" position="10,270" size="630,25" zPosition="3" transparent="1" >
            <convert type="TemplatedMultiContent">
                {"templates":
                    {"default": (23, [
                        MultiContentEntryText(pos = (0, 0),   size = (200, 25),  font = 0, flags = RT_HALIGN_LEFT,  color = 0xcccccc, text = 0), # name,
                        MultiContentEntryText(pos = (210, 0),  size = (200, 25),  font = 0, flags = RT_HALIGN_LEFT, color = 0xcccccc,  text = 1), # lang,
                        MultiContentEntryText(pos = (420, 0), size = (200, 25), font = 0, flags = RT_HALIGN_RIGHT, color = 0xcccccc, text = 2) # enabled,
                    ], False, "showOnDemand"),
                    },
                "fonts": [gFont("Regular", 18), gFont("Regular", 16)],
                "itemHeight": 23
                }
            </convert>
        </widget>
        <eLabel position="5,300" size="640,1" backgroundColor="#999999" />
        <widget source="providers" render="Listbox" scrollbarMode="showOnDemand" position="10,305" size="630,195" zPosition="3" transparent="1" >
            <convert type="TemplatedMultiContent">
                {"templates":
                    {"default": (23, [
                        MultiContentEntryText(pos = (0, 0),   size = (200, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 0), # name,
                        MultiContentEntryText(pos = (210, 0),  size = (200, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 1), # lang,
                        MultiContentEntryText(pos = (420, 0), size = (200, 25), font = 0, flags = RT_HALIGN_RIGHT, text = 2, color=0xFF000003) # enabled,
                    ], True, "showOnDemand"),
                    "notselected": (23, [
                        MultiContentEntryText(pos = (0, 0),   size = (200, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 0), # name,
                        MultiContentEntryText(pos = (210, 0),  size = (200, 25),  font = 0, flags = RT_HALIGN_LEFT,  text = 1), # lang,
                        MultiContentEntryText(pos = (420, 0), size = (200, 25), font = 0, flags = RT_HALIGN_RIGHT,  text = 2, color=0xFF000003) # enabled,
                    ], False, "showOnDemand")
                    },
                "fonts": [gFont("Regular", 18), gFont("Regular", 16)],
                "itemHeight": 23
                }
            </convert>
        </widget>
    </screen> """

    FOCUS_CONFIG, FOCUS_PROVIDERS = range(2)

    def __init__(self, session, searchSettings, seeker, streamed=False):
        Screen.__init__(self, session)
        ConfigListScreen.__init__(self, [], session=session)
        self.searchSettings = searchSettings
        self.searchParamsHelper = SearchParamsHelper(seeker, searchSettings)
        self.providers = seeker.seekers
        self.streamed = streamed
        self.focus = self.FOCUS_CONFIG
        self['providers'] = List([])
        self['providers_header'] = List([])
        self["key_green"] = Label(_("Save"))
        self["key_red"] = Label(_("Cancel"))
        self["key_blue"] = Label(_("Reset Defaults"))
        self["key_yellow"] = Label("")
        self["actions"] = ActionMap(["DirectionActions", "SetupActions", "OkCancelActions"],
        {
            "ok": self.keyOk,
            "cancel": self.keyCancel,
            "save":self.keySave,
            "up": self.keyUp,
            "down": self.keyDown,
            "right":self.keyRight,
            "left":self.keyLeft
        }, -2)
        self.onLayoutFinish.append(self.buildMenu)
        self.onLayoutFinish.append(self.updateProvidersHeader)
        self.onLayoutFinish.append(self.updateProvidersList)
        self.onLayoutFinish.append(self.setConfigFocus)

    def buildMenu(self):
        menuList = []
        menuList.append(getConfigListEntry(_("Preferred subtitles language") + ' 1', self.searchSettings.lang1))
        menuList.append(getConfigListEntry(_("Preferred subtitles language") + ' 2', self.searchSettings.lang2))
        menuList.append(getConfigListEntry(_("Preferred subtitles language") + ' 3', self.searchSettings.lang3))
        menuList.append(getConfigListEntry(_("Preffered Movie provider"), self.searchSettings.movieProvider))
        menuList.append(getConfigListEntry(_("Preffered TV show provider"), self.searchSettings.tvshowProvider))
        menuList.append(getConfigListEntry(_("Manual search"), self.searchSettings.manualSearch))
        menuList.append(getConfigListEntry(_("Subtitles provider timeout"), self.searchSettings.timeout))
        menuList.append(getConfigListEntry(_("Sort subtitles list by"), self.searchSettings.defaultSort))
        menuList.append(getConfigListEntry(_("Save subtitles as"), self.searchSettings.saveAs))
        menuList.append(getConfigListEntry(_("Add subtitle's language to filename"), self.searchSettings.addLangToSubsFilename))
        if not self.streamed:
            menuList.append(getConfigListEntry(_("Save subtitles to"), self.searchSettings.saveTo))
            if self.searchSettings.saveTo.value == 'custom':
                menuList.append(getConfigListEntry(_("Subtitles download path"), self.searchSettings.downloadPath))
        else:
            menuList.append(getConfigListEntry(_("Subtitles download path"), self.searchSettings.downloadPath))
        menuList.append(getConfigListEntry(_("Subtitles temp path"), self.searchSettings.tmpPath))
        menuList.append(getConfigListEntry(_("Always ask where to download subtitles"), self.searchSettings.askForDownloadLocation))
        menuList.append(getConfigListEntry(_("Always ask before overwriting existing subtitles"), self.searchSettings.askOverwriteExistingSubs))
        menuList.append(getConfigListEntry(_("Check search parameters before subtitles search"), self.searchSettings.openParamsDialogOnSearch))
        self["config"].setList(menuList)

    def updateProvidersHeader(self):
        self['providers_header'].list = [(_("Provider name"), _("Supported languages"), _("State")), ]

    def updateProvidersList(self):
        providerListGUI = []
        for provider in self.providers:
            providerName = provider.provider_name
            providerLangs = ','.join(provider.supported_langs)
            if provider.error is not None:
                providerState = _("error")
                providerStateColor = 0xff0000
            elif provider.settings_provider.getSetting('enabled'):
                providerState = _("enabled")
                providerStateColor = 0x00ff00
            else:
                providerState = _("disabled")
                providerStateColor = 0xffff00
            providerListGUI.append((providerName.encode('utf-8'), providerLangs, providerState, providerStateColor))
        self['providers'].list = providerListGUI

    def setConfigFocus(self):
        self.focus = self.FOCUS_CONFIG
        self['config'].instance.setSelectionEnable(True)
        self['providers'].style = 'notselected'

    def keyOk(self):
        if self.focus == self.FOCUS_PROVIDERS:
            provider = self.providers[self['providers'].index]
            if provider.error:
                self.showProviderError(provider)
            else: 
                self.openProviderSettings(provider)
        else:
            current = self['config'].getCurrent()[1]
            if current == self.searchSettings.downloadPath:
                currentPath = self.searchSettings.downloadPath.value
                self.session.openWithCallback(self.setDownloadPath, LocationBox, "", "", currentPath)
            elif current == self.searchSettings.tmpPath:
                currentPath = self.searchSettings.tmpPath.value
                self.session.openWithCallback(self.setTmpPath, LocationBox, "", "", currentPath)
            elif current in [self.searchSettings.lang1,
                                                    self.searchSettings.lang2,
                                                    self.searchSettings.lang3]:
                self.session.openWithCallback(self.setLanguage, MyLanguageSelection, current.value)

    def setLanguage(self, language=None):
        if language:
            self['config'].getCurrent()[1].value = language
            self.buildMenu()

    def setDownloadPath(self, downloadPath=None):
        if downloadPath:
            self.searchSettings.downloadPath.value = downloadPath
            self.buildMenu()

    def setTmpPath(self, tmpPath=None):
        if tmpPath:
            self.searchSettings.tmpPath.value = tmpPath
            self.buildMenu()

    def keySave(self):
        langChanged = (self.searchSettings.lang1.isChanged() or
                            self.searchSettings.lang2.isChanged() or
                            self.searchSettings.lang3.isChanged())
        for x in self["config"].list:
            x[1].save()
        self.close(langChanged)

    def keyCancel(self):
        for x in self["config"].list:
            x[1].cancel()
        self.close()

    def keyUp(self):
        if self.focus == self.FOCUS_CONFIG:
            self['config'].instance.moveSelection(self["config"].instance.moveUp)
        else:
            if self['providers'].index == 0:
                self.focus = self.FOCUS_CONFIG
                self['providers'].style = "notselected"
                self['config'].instance.setSelectionEnable(True)
                self['config'].setCurrentIndex(len(self['config'].list) - 1)
            else:
                self['providers'].selectPrevious()

    def keyDown(self):
        if self.focus == self.FOCUS_CONFIG:
            if self['config'].getCurrentIndex() != len(self['config'].list) - 1:
                self['config'].instance.moveSelection(self["config"].instance.moveDown)
            else:
                self.focus = self.FOCUS_PROVIDERS
                self['config'].instance.setSelectionEnable(False)
                self['providers'].style = 'default'
                self['providers'].index = 0
        else:
            self['providers'].selectNext()

    def keyRight(self):
        if self.focus == self.FOCUS_CONFIG:
            ConfigListScreen.keyRight(self)
            if self['config'].getCurrent()[1] in [self.searchSettings.saveTo]:
                self.buildMenu()

    def keyLeft(self):
        if self.focus == self.FOCUS_CONFIG:
            ConfigListScreen.keyLeft(self)
            if self['config'].getCurrent()[1] in [self.searchSettings.saveTo]:
                self.buildMenu()

    def showProviderError(self, provider):
        providerError = provider.error
        if isinstance(providerError, tuple):
            err_msg = providerError[1]
        else:
            err_msg = "unknown error"
            if isinstance(providerError,Exception):
                if isinstance(providerError, ImportError):
                    # No module named ...
                    err_msg= _("missing") + " python-%s "% (providerError.message.split()[-1]) + _("library")
                else:
                    err_msg = providerError.message
        msg = "%s: %s"%(provider.provider_name, err_msg)
        self.session.open(MessageBox, msg, MessageBox.TYPE_WARNING, timeout = 5)
    
    def openProviderSettings(self, provider):
        self.session.openWithCallback(self.openProviderSettingsCB, SubsSearchProviderMenu, provider)

    def openProviderSettingsCB(self, changed=False):
        if changed:
            self.updateProvidersList()
            self.searchParamsHelper.updateProviders()
            self.buildMenu()

class SubsSearchParamsMenu(Screen, ConfigListScreen):
    LIST_CONFIG = 0
    LIST_SUGGESTIONS = 1
    LIST_HISTORY = 2

    def __init__(self, session, seeker, searchSettings, titleList=None, resetSearchParams=True, enabledList=True, windowTitle=None):
        s = getDesktop(0).size()
        desktopSize = (s.width(), s.height())
        windowSize = (550, 330)
        self.skin = """
            <screen position="center,%d" size="550,330" >
                <widget source="sourceTitleInfo" render="Label" position="10,10" size="530,25" halign="center" font="Regular;22" foregroundColor="#66BFFF" />
                <widget source="sourceTitle" render="Label" position="10,35" size="530,55" halign="center" valign="center" font="Regular;21" />
                <widget source="searchTitleParams" render="Label" position="10,90" size="530,25" halign="center" font="Regular;22" foregroundColor="#ffbb00"/>
                <eLabel position="10,120" size="530,1" backgroundColor="#999999" />
                <widget name="config" position="10,130" size="530,160" scrollbarMode="showOnDemand" />
                <eLabel position="10,285" size="530,1" backgroundColor="#999999" />
                <widget source="pressOkTitleInfo" render="Label" position="10,295" size="530,25" halign="center" font="Regular;21" />
            </screen>""" % ((desktopSize[1] / 2) - (windowSize[1] / 2) + 50)


        Screen.__init__(self, session)
        ConfigListScreen.__init__(self, [], session=session)
        self["config"] = MyConfigList([], session, enabledList)
        if not self.handleInputHelpers in self["config"].onSelectionChanged:
            self["config"].onSelectionChanged.append(self.handleInputHelpers)
        self.searchParamsHelper = SearchParamsHelper(seeker, searchSettings)
        self.searchSettings = searchSettings
        if titleList is None:
            titleList = []
        self.sourceTitleList = titleList
        self.sourceTitle = titleList and titleList[0] or ""
        self.currentList = self.LIST_CONFIG
        self.windowTitle = windowTitle
        if len(self.sourceTitleList) == 0:
            self['sourceTitleInfo'] = StaticText(_("Source title not provided"))
        else:
            self['sourceTitleInfo'] = StaticText("%s [%d/%d]" % (_("Source title"), 1, len(self.sourceTitleList)))
        self['sourceTitle'] = StaticText(self.sourceTitle)
        self['searchTitleParams'] = StaticText(_("Search parameters"))
        self['pressOkTitleInfo'] = StaticText(_("Press OK to confirm"))
        self["suggestionActions"] = ActionMap(["DirectionActions", "ColorActions", "OkCancelActions"],
            {
                 "red": self.cancelToHistoryList,
                 "green": self.cancelToSuggestionsList,
                 "ok": self.switchToConfigList,
                 "cancel":self.cancelToConfigList,
                 "right": self.keyRight,
                 "rightRepeated": self.keyRight,
                 "left": self.keyLeft,
                 "leftRepeated": self.keyLeft,
                 "up": self.keyUp,
                 "upRepeated": self.keyUp,
                 "down": self.keyDown,
                 "downRepeated": self.keyDown
             }, -2)

        self["configActions"] = ActionMap(["OkCancelActions", "ColorActions", "DirectionActions"],
            {
                "ok": self.keyOK,
                "cancel": self.keyCancel,
                "green": self.switchToSuggestionsList,
                "red": self.switchToHistoryList,
                "blue": self.toggleSourceTitle,
                "right": self.keyRight,
                "rightRepeated": self.keyRight,
                "left": self.keyLeft,
                "leftRepeated": self.keyLeft,
                "up": self.keyUp,
                "upRepeated": self.keyUp,
                "down": self.keyDown,
                "downRepeated": self.keyDown
            }, -2)

        self['suggestionActions'].setEnabled(False)
        if resetSearchParams and titleList is not None:
            self.onLayoutFinish.append(self.detectSearchParams)
        self.onLayoutFinish.append(self.buildMenu)
        self.onLayoutFinish.append(self.setWindowTitle)
        self.onLayoutFinish.append(self.saveAll)
        self.onClose.append(self.removeSuggestionWindows)

    def setWindowTitle(self):
        if self.windowTitle is not None:
            self.setTitle(self.windowTitle)
        else: self.setTitle(_("Update Search params"))

    def buildMenu(self):
        menuList = []
        menuList.append(getConfigListEntry(_("Title") , self.searchSettings.title))
        menuList.append(getConfigListEntry(_("Type") , self.searchSettings.type))
        if self.searchSettings.type.value == "movie":
            menuList.append(getConfigListEntry(_("Year") , self.searchSettings.year))
        else:
            menuList.append(getConfigListEntry(_("Season") , self.searchSettings.season))
            menuList.append(getConfigListEntry(_("Episode") , self.searchSettings.episode))
        menuList.append(getConfigListEntry(_("Provider") , self.searchSettings.provider))
        menuList.append(getConfigListEntry(_("Use File path"), self.searchSettings.useFilePath))
        self["config"].list = menuList
        self["config"].setList(menuList)

    def detectSearchParams(self):
        self.searchParamsHelper.detectSearchParams(searchExpression=self.sourceTitle)
        self.searchParamsHelper.updateProviders()

    def toggleSourceTitle(self):
        if len(self.sourceTitleList) == 0:
            return
        currIdx = self.sourceTitleList.index(self.sourceTitle)
        if self.sourceTitle == self.sourceTitleList[-1]:
            currIdx = 0
        else: currIdx += 1
        self.sourceTitle = self.sourceTitleList[currIdx]
        self['sourceTitle'].text = self.sourceTitle
        self['sourceTitleInfo'].text = "%s [%d/%d]" % (_("Source title"), currIdx + 1, len(self.sourceTitleList))
        self.detectSearchParams()
        self.buildMenu()

    def switchToSuggestionsList(self):
        if not self['config'].enabled:
            return

        if self["config"].getCurrent()[1] == self.searchSettings.title:
            self["configActions"].setEnabled(False)
            self["suggestionActions"].setEnabled(True)
            self["config"].invalidateCurrent()
            self["config"].getCurrent()[1].enableHistory(False)
            if self["config"].getCurrent()[1].enableSuggestions(True):
                self.currentList = self.LIST_SUGGESTIONS
            else:
                self.cancelToConfigList()

    def switchToHistoryList(self):
        if not self['config'].enabled:
            return

        if self["config"].getCurrent()[1] == self.searchSettings.title:
            self["configActions"].setEnabled(False)
            self["suggestionActions"].setEnabled(True)
            self["config"].invalidateCurrent()
            self["config"].getCurrent()[1].enableSuggestions(False)
            if self["config"].getCurrent()[1].enableHistory(True):
                self.currentList = self.LIST_HISTORY
            else:
                self.cancelToConfigList()

    def switchToConfigList(self):
        self["config"].getCurrent()[1].enableSuggestions(False)
        self["config"].getCurrent()[1].enableHistory(False)
        self["suggestionActions"].setEnabled(False)
        self["configActions"].setEnabled(True)
        self.currentList = self.LIST_CONFIG

    def cancelToHistoryList(self):
        self["config"].invalidateCurrent()
        self["config"].getCurrent()[1].cancelSuggestions()
        self.switchToHistoryList()

    def cancelToSuggestionsList(self):
        self["config"].invalidateCurrent()
        self["config"].getCurrent()[1].cancelSuggestions()
        self.switchToSuggestionsList()

    def cancelToConfigList(self):
        self["config"].invalidateCurrent()
        self["config"].getCurrent()[1].cancelSuggestions()
        self.switchToConfigList()

    def addToHistory(self):
        history = self.searchSettings.history.value.split(',')
        if history[0] == '':
            del history[0]
        if self.searchSettings.title.value in history:
            history.remove((self.searchSettings.title.value))
        history.insert(0, (self.searchSettings.title.value))
        if len(history) == 30:
            history.pop()
        self.searchSettings.history.value = ",".join(history)
        self.searchSettings.history.save()

    def keySave(self):
        for x in self["config"].list:
            x[1].save()
        self.close(True)

    def keyCancel(self):
        for x in self["config"].list:
            x[1].cancel()
        self.close()

    def keyOK(self):
        if self.currentList == self.LIST_CONFIG:
            self.addToHistory()
            self.saveAll()
            self.close(True)
        elif self.currentList in (self.LIST_SUGGESTIONS, self.LIST_HISTORY):
            self.switchToConfigList()

    def keyDown(self):
        if not self['config'].enabled:
            self['config'].enableList()
        elif self.currentList in (self.LIST_SUGGESTIONS, self.LIST_HISTORY):
            self["config"].getCurrent()[1].currentListDown()
            self["config"].invalidateCurrent()
        elif self.currentList == self.LIST_CONFIG:
            self['config'].instance.moveSelection(self["config"].instance.moveDown)

    def keyUp(self):
        if not self['config'].enabled:
            self['config'].enableList()
        elif self.currentList in (self.LIST_SUGGESTIONS, self.LIST_HISTORY):
            self["config"].getCurrent()[1].currentListUp()
            self["config"].invalidateCurrent()
        elif self.currentList == self.LIST_CONFIG:
            self['config'].instance.moveSelection(self["config"].instance.moveUp)

    def keyLeft(self):
        if not self['config'].enabled:
            self['config'].enableList()
        elif self.currentList in (self.LIST_SUGGESTIONS, self.LIST_HISTORY):
            self["config"].getCurrent()[1].currentListPageUp()
            self["config"].invalidateCurrent()
        elif self.currentList == self.LIST_CONFIG:
            ConfigListScreen.keyLeft(self)
            if self['config'].getCurrent()[1] == self.searchSettings.type:
                self.searchParamsHelper.updateProviders()
                self.buildMenu()

    def keyRight(self):
        if not self['config'].enabled:
            self['config'].enableList()
        elif self.currentList in (self.LIST_SUGGESTIONS, self.LIST_HISTORY):
            self["config"].getCurrent()[1].currentListPageDown()
            self["config"].invalidateCurrent()
        elif self.currentList == self.LIST_CONFIG:
            ConfigListScreen.keyRight(self)
            if self['config'].getCurrent()[1] == self.searchSettings.type:
                self.searchParamsHelper.updateProviders()
                self.buildMenu()

    def removeSuggestionWindows(self):
        if hasattr(self.searchSettings.title, 'suggestionsWindow'):
            suggestionsWindow = self.searchSettings.title.suggestionsWindow
            if suggestionsWindow is not None:
                self.session.deleteDialog(suggestionsWindow)
                self.searchSettings.title.suggestionsWindow = None
        if hasattr(self.searchSettings.title, 'historyWindow'):
            historyWindow = self.searchSettings.title.historyWindow
            if historyWindow is not None:
                self.session.deleteDialog(historyWindow)
                self.searchSettings.title.historyWindow = None


class SubsSearchProviderMenu(Screen, ConfigListScreen):
    skin = """
            <screen position="center,center" size="610,435" >
                <widget name="key_red" position="10,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#9f1313" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_green" position="160,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#1f771f" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_yellow" position="310,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#a08500" shadowOffset="-2,-2" shadowColor="black" />
                <widget name="key_blue" position="460,5" zPosition="1" size="140,45" font="Regular;20" halign="center" valign="center" backgroundColor="#18188b" shadowOffset="-2,-2" shadowColor="black" />
                <eLabel position="-1,55" size="612,1" backgroundColor="#999999" />
                <widget name="config" position="0,75" size="610,360" scrollbarMode="showOnDemand" />
            </screen>"""

    def __init__(self, session, provider):
        Screen.__init__(self, session)
        self.provider = provider
        ConfigListScreen.__init__(self, [], session=session)
        self["actions"] = ActionMap(["SetupActions", "ColorActions"],
            {
                "cancel": self.keyCancel,
                "green": self.keySave,
                "red": self.keyCancel,
                "blue": self.resetDefaults,
            }, -2)

        self["key_green"] = Label(_("Save"))
        self["key_red"] = Label(_("Cancel"))
        self["key_blue"] = Label(_("Reset Defaults"))
        self["key_yellow"] = Label("")
        self.buildMenu()
        self.onLayoutFinish.append(self.updateTitle)

    def updateTitle(self):
        self.setTitle(self.provider.provider_name.encode('utf-8') + _("settings"))

    def buildMenu(self):
        settingsProvider = self.provider.settings_provider
        settingList = settingsProvider.getE2Settings()
        self["config"].list = settingList
        self["config"].setList(settingList)

    def resetDefaults(self):
        for x in self["config"].list:
            x[1].value = x[1].default
        self.buildMenu()

    def keySave(self):
        for x in self["config"].list:
            x[1].save()
        configfile.save()
        self.close(True)

    def keyCancel(self):
        for x in self["config"].list:
            x[1].cancel()
        self.close()

    def keyLeft(self):
        ConfigListScreen.keyLeft(self)

    def keyRight(self):
        ConfigListScreen.keyRight(self)
