#
#  MerlinUpdateNotifier E2 Plugin
#
#  $Id: plugin.py,v 1.1 2010-11-12 21:12:57 weazle Exp $
#
#  Idea by weazle (c) 2010
#  Coding support by Dr. Best
#  But most of the bad code was written by Shaderman :)
#  Support: www.dreambox-tools.info
#
#  This plugin is licensed under the Creative Commons 
#  Attribution-NonCommercial-ShareAlike 3.0 Unported 
#  License. To view a copy of this license, visit
#  http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative
#  Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
#
#  Alternatively, this plugin may be distributed and executed on hardware which
#  is licensed by Dream Multimedia GmbH.

#  This plugin is NOT free software. It is open source, you are allowed to
#  modify it (if you keep the license), but it may not be commercially 
#  distributed other than under the conditions noted above.
#

from Plugins.Plugin import PluginDescriptor
from Screens.Screen import Screen
from Components.ActionMap import ActionMap
from Components.Sources.StaticText import StaticText
from Components.config import config, ConfigSelection, getConfigListEntry, ConfigSubsection, ConfigClock, ConfigInteger, NoSave, ConfigBoolean
from Components.ConfigList import ConfigListScreen
from Components.Ipkg import IpkgComponent
from enigma import eDVBLocalTimeHandler
from Components.Label import Label
from Screens.MessageBox import MessageBox
from time import time, strftime, localtime, mktime
from datetime import datetime, timedelta
from timer import Timer, TimerEntry
from Tools import Notifications
from Plugins.SystemPlugins.SoftwareManager.plugin import UpdatePlugin

# for localized messages
from . import _


config.plugins.merlin = ConfigSubsection()
config.plugins.merlin.notifierInterval = ConfigSelection(default = "-1", choices = [("-1",_("disabled")),("1",_("on Mondays")),("2",_("on Tuesdays")), ("4",_("on Wednesdays")), ("8",_("on Thursdays")), ("16",_("on Fridays")), ("32",_("on Saturdays")), ("64",_("on Sundays")), ("127",_("daily"))])
config.plugins.merlin.notifierTime = ConfigClock(default = 18000) # 06:00am
config.plugins.merlin.notifierLastUpdate = ConfigInteger(default = 0)
config.plugins.merlin.notifierNextUpdate = ConfigInteger(default = 0)
config.plugins.merlin.notifierSettingsChanged = NoSave(ConfigBoolean())

Notifications.notificationQueue.registerDomain("MerlinUpdateCheck", _("MerlinUpdateCheck"), Notifications.ICON_DEFAULT)


class UpgradableListIpkgComponent(IpkgComponent):

	EVENT_CACHING = 13
	CMD_LIST_UPGRADABLE = 5
	
	def startCmd(self, cmd, args = None):
		if cmd == self.CMD_LIST_UPGRADABLE:
			self.fetchedList = []
			self.runCmd("list-upgradable")	
			self.setCurrentCommand(cmd)		
		else:
			IpkgComponent.startCmd(self, cmd, args)

	def parseLine(self, data):
		self.callCallbacks(self.EVENT_CACHING, data)
		if self.currentCommand == self.CMD_LIST_UPGRADABLE:
			item = data.split(' - ', 2)
			self.fetchedList.append(item)
			self.callCallbacks(self.EVENT_LISTITEM, item)
		else:
			if data.find('opkg:') == 0:
				self.callCallbacks(self.EVENT_ERROR, None)
			if data.find('Collected errors:') == 0:
				self.callCallbacks(self.EVENT_ERROR, None)
			else:
				IpkgComponent.parseLine(self, data)

class MerlinUpdateCheck(object):

	ERROR_LOG_FILENAME = "/tmp/update_error.txt"

	def __init__(self, session, ipkgstatuscallback = None):
		self.session = session
		self.cmd = UpgradableListIpkgComponent.CMD_UPDATE
		self.error = 0
		self.ipkgData = ""
		self.ipkgStatusCallback = ipkgstatuscallback

	def sendCallback(self,text):
		if self.ipkgStatusCallback is not None:
			self.ipkgStatusCallback(text)

	def checkForUpdate(self, cmd = UpgradableListIpkgComponent.CMD_UPDATE):
		self.error = 0
		self.cmd = cmd
		if cmd == UpgradableListIpkgComponent.CMD_UPDATE:
			self.sendCallback("opkg update...")
		else:
			self.sendCallback("opkg list-upgradable...")
		self.ipkg = UpgradableListIpkgComponent()
		self.ipkg.addCallback(self.ipkgCallback)
		self.ipkg.startCmd(self.cmd)

	def ipkgCallback(self, event, param):
		if event == UpgradableListIpkgComponent.EVENT_CACHING:
			self.ipkgData += param + "\n"
		elif event == UpgradableListIpkgComponent.EVENT_ERROR:
			self.error += 1
		elif event == UpgradableListIpkgComponent.EVENT_DONE:
			if self.cmd == UpgradableListIpkgComponent.CMD_UPDATE:
				self.checkForUpdate(UpgradableListIpkgComponent.CMD_LIST_UPGRADABLE)
			elif self.cmd == UpgradableListIpkgComponent.CMD_LIST_UPGRADABLE:
				config.plugins.merlin.notifierLastUpdate.setValue(int(time()))
				config.plugins.merlin.notifierLastUpdate.save()
				if self.error == 0:
					if len(self.ipkg.fetchedList):
						self.sendCallback(_("There are updates available on the Merlin feeds..."))
						if isinstance(self, MerlinUpdateNotifier):
							self.session.openWithCallback(self.runUpgrade, MessageBox, _("There are updates available on the Merlin feeds. Check www.dreambox-tools.info for details. Do you want to update your Merlin image?\nPlease wait a moment after starting the update!"), default = True)
						else:
							Notifications.AddNotificationWithCallback(self.runUpgrade, MessageBox, _("There are updates available on the Merlin feeds. Check www.dreambox-tools.info for details. Do you want to update your Merlin image?\nPlease wait a moment after starting the update!"), default = False, domain = "MerlinUpdateCheck")
					else:
						self.sendCallback(_("No updates available..."))
						print "[MerlinUpdateCheck] No updates available"
						if isinstance(self, MerlinUpdateNotifier):
							self.session.open(MessageBox, _("There are no updates available on the Merlin feeds.\nCheck www.dreambox-tools.info for details."), MessageBox.TYPE_INFO)
				else:
					self.updateError()
				self.ipkgData = ""

	# save some information in a text file on errors
	def saveErrorFile(self):
		print "[MerlinUpdateCheck] Saving error to file..."
		try:
			f = open(MerlinUpdateCheck.ERROR_LOG_FILENAME, "w")
			f.write("COMMAND:\n" + self.command + "\n\nERROR:\n" + self.ipkgData)
			f.close()
		except:
			pass

	# on of the "opkg" commands raised an error...
	def updateError(self):
		self.sendCallback(_("There was an error..."))
		print "[MerlinUpdateCheck] There was an error..."
		self.saveErrorFile()
		if isinstance(self, MerlinUpdateNotifier):
			self.session.open(MessageBox, _("There was a problem running <%s>.\nCommand output was written to %s. Please visit www.dreambox-tools.info and report this error.") % (self.command, MerlinUpdateCheck.ERROR_LOG_FILENAME), MessageBox.TYPE_ERROR)
		else:
			Notifications.AddNotification(MessageBox, _("There was a problem running <%s>.\nCommand output was written to %s. Please visit www.dreambox-tools.info and report this error.") % (self.command, MerlinUpdateCheck.ERROR_LOG_FILENAME), MessageBox.TYPE_ERROR, domain = "MerlinUpdateCheck")

	# the user wants to install updates
	def runUpgrade(self, result):
		if result:
			self.session.open(UpdatePlugin)
		self.sendCallback("")

class MerlinUpdateNotifier(Screen, ConfigListScreen, MerlinUpdateCheck):

	skin = """
		<screen position="center,center" size="560,320" >
			<ePixmap pixmap="skin_default/buttons/red.png" position="0,0" zPosition="0" size="140,40" transparent="1" alphatest="on" />
			<ePixmap pixmap="skin_default/buttons/green.png" position="140,0" zPosition="0" size="140,40" transparent="1" alphatest="on" />
			<ePixmap pixmap="skin_default/buttons/yellow.png" position="280,0" zPosition="0" size="140,40" transparent="1" alphatest="on" />
			<ePixmap pixmap="skin_default/buttons/blue.png" position="420,0" zPosition="0" size="140,40" transparent="1" alphatest="on" />
			<widget render="Label" source="key_red" position="0,0" size="140,40" zPosition="5" valign="center" halign="center" backgroundColor="red" font="Regular;21" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-1,-1" />
			<widget render="Label" source="key_green" position="140,0" size="140,40" zPosition="5" valign="center" halign="center" backgroundColor="red" font="Regular;21" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-1,-1" />
			<widget render="Label" source="key_yellow" position="280,0" size="140,40" zPosition="5" valign="center" halign="center" backgroundColor="red" font="Regular;21" transparent="1" foregroundColor="white" shadowColor="black" shadowOffset="-1,-1" />
			<widget name="config" position="20,50" size="520,150" scrollbarMode="showOnDemand" />
			<widget render="Label" source="lastCheck" position="20,215" size="160,25" font="Regular;20" />
			<widget render="Label" source="lastCheckTime" position="180,215" size="260,25" font="Regular;20" />
			<widget render="Label" source="nextCheck" position="20,240" size="160,25" font="Regular;20" />
			<widget render="Label" source="nextCheckTime" position="180,240" size="260,25" font="Regular;20" />
			<widget render="Label" source="infotext" position="20,275" size="520,25" font="Regular;20" />
		</screen>"""

	def __init__(self, session):
		self.session = session
		Screen.__init__(self, session)
		MerlinUpdateCheck.__init__(self, session, self.ipkgStatusCallback)
		self["key_red"] = StaticText(_("Exit"))
		self["key_green"] = StaticText(_("Save"))
		self["key_yellow"] = StaticText(_("Check now"))
		
		self["lastCheck"] = StaticText(_("Last run:"))
		self["lastCheckTime"] = StaticText("-")
		self["nextCheck"] = StaticText(_("Next run:"))
		self["nextCheckTime"] = StaticText("-")

		self["infotext"] = StaticText("")

		self.list = []
		self.createSetup()
		ConfigListScreen.__init__(self, self.list)
		self.addNotifiers()

		self["setupActions"] = ActionMap(["SetupActions", "ColorActions"],
		{
			"ok": self.save,
			"green": self.save,
			"yellow": self.checkNow,
			"red": self.exit,
			"cancel": self.exit,
		}, -1)

		self.onLayoutFinish.append(self.setCustomTitle)
		self.onShown.append(self.updateStatusText)

	# set the screen title
	def setCustomTitle(self):
		self.onLayoutFinish.remove(self.setCustomTitle)
		self.setTitle(_("Merlin Update Notifier"))

	# create the ConfigList list
	def createSetup(self):
		list = []
		list.append(getConfigListEntry(_("Check for updates"), config.plugins.merlin.notifierInterval))
		if int(config.plugins.merlin.notifierInterval.getValue()) > -1:
			list.extend((
				getConfigListEntry(_("Update time"), config.plugins.merlin.notifierTime),
			))
		self.list = list

	# user changed the interval
	def settingsChanged(self, configElement = None):
		print "[MerlinUpdateNotifier] settings changed..."
		self.createSetup()
		self["config"].setList(self.list)
		self.updateStatusText()

	# status-callback from MerlinUpdateCheck
	def ipkgStatusCallback(self, text):
		self["infotext"].text = text

	# update the times for last and next update
	def updateStatusText(self):
		nextUpdate = int(config.plugins.merlin.notifierNextUpdate.getValue())
		if nextUpdate == 0:
			self["nextCheckTime"].text = ("-")
		else:
			t = localtime(nextUpdate)
			self["nextCheckTime"].text = strftime("%02d.%02d.%04d, %02d:%02d" % (t[2], t[1], t[0], t[3], t[4]))
			
		lastUpdate = config.plugins.merlin.notifierLastUpdate.getValue()
		if lastUpdate != 0:
			t = localtime(lastUpdate)
			self["lastCheckTime"].text = strftime("%02d.%02d.%04d, %02d:%02d" % (t[2], t[1], t[0], t[3], t[4]))
		else: # never checked before
			self["lastCheckTime"].text = ("-")

	# save the settings and update the status text
	def save(self):
		print "[MerlinUpdateNotifier] saving settings..."
		# notify the MerlinUpdateCheckTimer
		config.plugins.merlin.notifierSettingsChanged.setValue(not config.plugins.merlin.notifierSettingsChanged.getValue())
		if int(config.plugins.merlin.notifierInterval.getValue()) == -1:
			config.plugins.merlin.notifierNextUpdate.setValue(0)
		else:
			config.plugins.merlin.notifierNextUpdate.save()
		for x in self["config"].list:
			x[1].save()
		self.updateStatusText()

	# exit the plugin
	def exit(self):
		for x in self["config"].list:
			x[1].cancel()
		self.removeNotifiers()
		self.close()

	def addNotifiers(self):
		config.plugins.merlin.notifierInterval.addNotifier(self.settingsChanged, initial_call = False)

	def removeNotifiers(self):
		config.plugins.merlin.notifierInterval.removeNotifier(self.settingsChanged)

	# MerlinUpdateCheck function
	def checkNow(self):
		self.checkForUpdate()
		self.updateStatusText()

class UpdateTimerEntry(MerlinUpdateCheck, TimerEntry):
	def __init__(self, session, begin, end):
		self.session = session
		MerlinUpdateCheck.__init__(self, session)
		TimerEntry.__init__(self, begin, end)

	# TimerEntry function
	def getNextActivation(self):
		return self.begin

	# TimerEntry function
	def activate(self):
		if self.state == self.StateRunning and not self.disabled:
			print "[UpdateTimerEntry] it's time to check for updates..."
			self.checkForUpdate() # MerlinUpdateCheck function
		return True

	# TimerEntry function
	def shouldSkip(self):
		return False

	# Timer function
	def timeChanged(self):
		print "[UpdateTimerEntry] update time changed..."
		if int(config.plugins.merlin.notifierInterval.getValue()) != -1:
			config.plugins.merlin.notifierNextUpdate.setValue(self.begin)
			config.plugins.merlin.notifierNextUpdate.save()

class MerlinUpdateCheckTimer(MerlinUpdateCheck, Timer):
	def __init__(self, session):
		self.session = session
		MerlinUpdateCheck.__init__(self, session)
		Timer.__init__(self)
		print "[MerlinUpdateCheckTimer] Starting..."

		# let's wait for the system time being up to date before starting the timers. needed when the box was powered off
		if not eDVBLocalTimeHandler.getInstance().ready():
			self.local_time_handler_conn = eDVBLocalTimeHandler.getInstance().m_timeUpdated.connect(self.run)
		else:
			self.addNotifierTimer()

		self.addNotifiers()

	# we're starting delayed to have the transponder time after booting
	def run(self):
		self.local_time_handler_conn = None
		self.addNotifierTimer()

	# get the begin time for our timer
	def getTimerBegin(self):
		now = localtime(int(time()))
		dt = datetime(now.tm_year, now.tm_mon, now.tm_mday, config.plugins.merlin.notifierTime.getValue()[0], config.plugins.merlin.notifierTime.getValue()[1])
		begin = int(mktime(dt.timetuple()))
		return begin

	# set the interval for our repeating timer
	def setTimerInterval(self):
		print "[MerlinUpdateCheckTimer] timer interval changed..."
		interval = int(config.plugins.merlin.notifierInterval.value)
		if interval == -1: # disabled
			self.timerEntry.disabled = True
		else: # daily, or weekly (monday - sunday)
			self.timerEntry.disabled = False
			self.timerEntry.repeated = interval

	# settings were changed
	def addNotifierTimer(self):
		print "[MerlinUpdateCheckTimer] Adding timer..."
		nextUpdate = int(config.plugins.merlin.notifierNextUpdate.getValue())
		now = int(time())
		if nextUpdate != 0 and now > nextUpdate: # do we need to catch up?
			self.checkForUpdate()

		begin = self.getTimerBegin()
		end = begin -1
		self.timerEntry = UpdateTimerEntry(self.session, begin, end)
		self.setTimerInterval()
		self.addTimerEntry(self.timerEntry) # Timer class function
		# at this point, the next update time is calculated - save it
		self.saveUpdateValues(self.timerEntry.begin)

	# called when user saves changes
	def updateNotifierTimer(self, configElement = None):
		print "[MerlinUpdateCheckTimer] updating the timer..."
		# get our timer object back on the road. doesn't look nice, but it works :)
		if len(self.timer_list):
			self.timerEntry = self.timer_list[0]
		else:
			self.timerEntry = self.processed_timers[0]

		self.timerEntry.begin = self.getTimerBegin()
		self.timerEntry.end = self.timerEntry.begin -1
		self.setTimerInterval()
		self.timeChanged(self.timerEntry) # Timer class function
		self.saveUpdateValues(self.timerEntry.begin)

	# save the next update time, it'll be shown in the GUI
	def saveUpdateValues(self, begin):
		if int(config.plugins.merlin.notifierInterval.getValue()) == -1:
			config.plugins.merlin.notifierNextUpdate.setValue(0)
		else:
			config.plugins.merlin.notifierNextUpdate.setValue(begin)
		config.plugins.merlin.notifierNextUpdate.save()

	def addNotifiers(self):
		config.plugins.merlin.notifierSettingsChanged.addNotifier(self.updateNotifierTimer, initial_call = False)

def sessionstart(session, **kwargs):
	MerlinUpdateCheckTimer(session)

def openMerlinUpdateNotifier(session, **kwargs):
	session.open(MerlinUpdateNotifier)

def startMerlinUpdateNotifier(menuid):
	if menuid != "merlin":
		return [ ]
	return [(_("Merlin Update Notifier"), openMerlinUpdateNotifier, "MerlinUpdateNotifier", None)]

def Plugins(**kwargs):
	return [PluginDescriptor(where = [PluginDescriptor.WHERE_SESSIONSTART], fnc = sessionstart), PluginDescriptor(name="MerlinUpdateNotifier", description=_("Merlin Update Notifier"), where = PluginDescriptor.WHERE_MENU, fnc=startMerlinUpdateNotifier) ]
